package com.biz.crm.tpm.business.marketing.strategy.local.repository;


import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.util.ObjectConvertStringUtil;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategyBudget;
import com.biz.crm.tpm.business.marketing.strategy.local.mapper.MarketingStrategyBudgetMapper;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyBudgetDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyItemDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyBudgetVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.beans.PropertyDescriptor;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 营销策略(MarketingStrategy)表数据库访问层
 *
 * @author wanghaojia
 * @since 2022-11-05 16:06:03
 */
@Component
public class MarketingStrategyBudgetRepository extends ServiceImpl<MarketingStrategyBudgetMapper, MarketingStrategyBudget> {

    @Autowired(required = false)
    private MarketingStrategyBudgetMapper marketingStrategyBudgetMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoListProperties(List<MarketingStrategyBudgetVo> voList) {
        if (CollectionUtils.isEmpty(voList)){
            return;
        }
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(MarketingStrategyBudgetVo.class);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors).collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        for (MarketingStrategyBudgetVo item : voList) {
            this.fillVoProperties(item,propertyMap);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(MarketingStrategyBudgetVo item) {
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(MarketingStrategyBudgetVo.class);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors).collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        this.fillVoProperties(item,propertyMap);
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(MarketingStrategyBudgetVo item,Map<String, PropertyDescriptor> propertyMap) {
        if (null == item){
            return;
        }
        //统一做BigDecimal,integer,Date类型格式转换
        ObjectConvertStringUtil.fillObjectStrProperties(item, MarketingStrategyBudgetVo.class,propertyMap);
        BigDecimal useAmount = Optional.ofNullable(item.getUseAmount()).orElse(BigDecimal.ZERO);
        BigDecimal usedAmount = Optional.ofNullable(item.getUsedAmount()).orElse(BigDecimal.ZERO);
        item.setUsedAmount(usedAmount);
        item.setUsableAmount(useAmount.subtract(usedAmount));
    }

    /**
     * 根据测量编码查询所有关联预算
     * @param strategyCode 策略编码
     */
    public List<MarketingStrategyBudgetDto> findDtoListByStrategyCode(String strategyCode) {
        if (StringUtils.isBlank(strategyCode)){
            return Lists.newArrayList();
        }
        LambdaQueryWrapper<MarketingStrategyBudget> wrapper = Wrappers.lambdaQuery(MarketingStrategyBudget.class)
                .eq(MarketingStrategyBudget::getStrategyCode,strategyCode)
                .eq(MarketingStrategyBudget::getTenantCode, TenantUtils.getTenantCode())
                .eq(MarketingStrategyBudget::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
        List<MarketingStrategyBudget> list = this.list(wrapper);
        if (CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        List<MarketingStrategyBudgetDto> dtoList = (List<MarketingStrategyBudgetDto>) nebulaToolkitService.copyCollectionByWhiteList(list, MarketingStrategyBudget.class, MarketingStrategyBudgetDto.class, HashSet.class, ArrayList.class);
        return dtoList;
    }

    /**
     * 根据细案编码查询
     **/
    public List<MarketingStrategyBudgetDto> listByStrategyItem(MarketingStrategyItemDto dto) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(dto.getStrategyCode())
                && CollectionUtil.isEmpty(dto.getStrategyCodeList())
                && CollectionUtil.isEmpty(dto.getStrategyItemCodeList())) {
            return Lists.newArrayList();
        }
        return marketingStrategyBudgetMapper.listDtoList(dto);
    }
    /**
     * 根据细案编码查询
     **/
    public List<MarketingStrategyBudgetDto> listByStrategyCode(String strategyCode) {
        if (org.apache.commons.lang3.StringUtils.isEmpty(strategyCode)) {
            return Lists.newArrayList();
        }
        MarketingStrategyItemDto itemDto = new MarketingStrategyItemDto();
        itemDto.setStrategyCode(strategyCode);
        return listByStrategyItem(itemDto);
    }

    /**
     * 根据细案编码查询
     **/
    public List<MarketingStrategyBudget> listByStrategyCodeList(List<String> strategyCodeList) {
        if (CollectionUtils.isEmpty(strategyCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(MarketingStrategyBudget::getStrategyCode, strategyCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }
    /**
     * 根据细案编码查询
     **/
    public List<MarketingStrategyBudget> listByStrategyItemCodeList(List<String> strategyItemCodeList) {
        if (CollectionUtils.isEmpty(strategyItemCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(MarketingStrategyBudget::getStrategyItemCode, strategyItemCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    public void deleteByIds(List<String> ids) {
        if (org.springframework.util.CollectionUtils.isEmpty(ids)){
            return;
        }
        LambdaUpdateWrapper<MarketingStrategyBudget> updateWrapper = new UpdateWrapper<MarketingStrategyBudget>().lambda();
        updateWrapper.set(MarketingStrategyBudget::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(MarketingStrategyBudget::getId, ids);
        this.update(updateWrapper);
    }

    public void deleteByStrategyCodes(List<String> strategyCodes) {
        if (CollectionUtils.isEmpty(strategyCodes)){
            return;
        }
        LambdaUpdateWrapper<MarketingStrategyBudget> updateWrapper = new UpdateWrapper<MarketingStrategyBudget>().lambda();
        updateWrapper.set(MarketingStrategyBudget::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(MarketingStrategyBudget::getStrategyCode, strategyCodes);
        updateWrapper.eq(MarketingStrategyBudget::getTenantCode, TenantUtils.getTenantCode());
        this.update(updateWrapper);
    }

    public void deleteByStrategyItemCodes(List<String> strategyItemCodes) {
        if (CollectionUtils.isEmpty(strategyItemCodes)){
            return;
        }
        LambdaUpdateWrapper<MarketingStrategyBudget> updateWrapper = new UpdateWrapper<MarketingStrategyBudget>().lambda();
        updateWrapper.set(MarketingStrategyBudget::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(MarketingStrategyBudget::getStrategyItemCode, strategyItemCodes);
        updateWrapper.eq(MarketingStrategyBudget::getTenantCode, TenantUtils.getTenantCode());
        this.update(updateWrapper);
    }

    public void deletePhysicalByStrategyCode(String strategyCode) {
        if (StringUtils.isBlank(strategyCode)){
            return;
        }
        LambdaQueryWrapper<MarketingStrategyBudget> updateWrapper = new LambdaQueryWrapper<MarketingStrategyBudget>();
//        updateWrapper.set(MarketingStrategyBudget::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.eq(MarketingStrategyBudget::getStrategyCode, strategyCode);
        updateWrapper.eq(MarketingStrategyBudget::getTenantCode, TenantUtils.getTenantCode());
        this.remove(updateWrapper);
    }

    /**
     * 查询营销策略预算信息
     */
    public List<MarketingStrategyBudgetDto> findListByConditions(MarketingStrategyBudgetDto dto) {
        return marketingStrategyBudgetMapper.findListByConditions(dto);
    }

    /**
     * 操作预算，根据行项目编码+预算编码列举数据
     * @param operateCodeList 操作编码
     */
    public List<MarketingStrategyBudgetDto> listBudgetInfoByOperateList(List<String> operateCodeList) {
        return marketingStrategyBudgetMapper.listBudgetInfoByOperateList(operateCodeList);
    }

    public Page<MarketingStrategyBudgetVo> findByConditions(Pageable pageable, MarketingStrategyBudgetDto dto) {
        Page<MarketingStrategyBudgetVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<MarketingStrategyBudgetVo> pageList = this.marketingStrategyBudgetMapper.findConditions(page, dto);
        fillVoListProperties(pageList.getRecords());
        return pageList;
    }

    /**
     * 根据预算表的数据更新策略头表的使用金额
     * @param strategyCodeList 策略编码
     */
    public void updateStrategyUsedAmount(List<String> strategyCodeList) {
        baseMapper.updateStrategyUsedAmount(strategyCodeList);
    }

    public void updateHeadStatus(List<String> strategyItemCodeList) {
        if (CollectionUtils.isEmpty(strategyItemCodeList)){
            return;
        }
        baseMapper.updateHeadStatus(strategyItemCodeList);
    }
}

