package com.biz.crm.tpm.business.marketing.strategy.local.repository;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.util.NumberStringDealUtil;
import com.biz.crm.mn.common.base.util.ObjectConvertStringUtil;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategyItem;
import com.biz.crm.tpm.business.marketing.strategy.local.mapper.MarketingStrategyItemMapper;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyBudgetDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyItemDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.pojo.MarketingStrategyItemScope;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketStrategyItemExportVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyItemVo;
import com.biz.crm.tpm.business.month.budget.sdk.constant.MonthBudgetConstant;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;


/**
 * 营销策略(MarketingStrategy)表数据库访问层
 *
 * @author wanghaojia
 * @since 2022-11-03 18:22:18
 */
@Component
public class MarketingStrategyItemRepository extends ServiceImpl<MarketingStrategyItemMapper, MarketingStrategyItem> {

    @Autowired(required = false)
    private MarketingStrategyItemMapper marketingStrategyItemMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private MarketingStrategyBudgetRepository marketingStrategyBudgetRepository;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    /**
     * 按条件查询营销策略
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     */
    public Page<MarketingStrategyItemVo> findByConditions(Pageable pageable, MarketingStrategyItemDto dto) {
        Page<MarketingStrategyItemVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<MarketingStrategyItemVo> pageList = this.marketingStrategyItemMapper.findByConditions(page, dto);
        fillVoListProperties(pageList.getRecords());
        return pageList;
    }

    /**
     * 查策略明细信息及其附加其他信息
     * @param strategyCode 策略编码
     * @return
     */
    public List<MarketingStrategyItemDto> findDtoAndAttachListByStrategyCode(String strategyCode){
        MarketingStrategyItemDto dto = new MarketingStrategyItemDto();
        dto.setStrategyCode(strategyCode);
        return findDtoAndAttachList(dto);
    }

    /**
     * 查策略明细信息及其附加其他信息
     */
    public List<MarketingStrategyItemDto> findDtoAndAttachList(MarketingStrategyItemDto dto){
        List<MarketingStrategyItemDto> dtoList = findDtoList(dto);
        //营销策略就附带一个预算
        List<MarketingStrategyBudgetDto> budgetList = marketingStrategyBudgetRepository.listByStrategyItem(dto);
        if (!CollectionUtils.isEmpty(budgetList)){
            convertMonthBudgetProperty(budgetList);
            for (MarketingStrategyBudgetDto budgetDto : budgetList) {
                NumberStringDealUtil.setStringIfNotNull(budgetDto.getUseAmount(), budgetDto::setUseAmountStr);
                BigDecimal afterFreezeAmount = Optional.ofNullable(budgetDto.getAfterFreezeAmount()).orElse(BigDecimal.ZERO);
                BigDecimal usedStrategyAmount = Optional.ofNullable(budgetDto.getUsedStrategyAmount()).orElse(BigDecimal.ZERO);
                BigDecimal usableStrategyAmount = afterFreezeAmount.subtract(usedStrategyAmount);
                budgetDto.setUsableStrategyAmount(usableStrategyAmount);
            }
            Map<String, List<MarketingStrategyBudgetDto>> budgetMap = budgetList.stream().filter(item -> StringUtils.isNotBlank(item.getStrategyItemCode())).collect(Collectors.groupingBy(MarketingStrategyBudgetDto::getStrategyItemCode));
            for (MarketingStrategyItemDto marketingStrategyItemDto : dtoList) {
                if (budgetMap.containsKey(marketingStrategyItemDto.getStrategyItemCode())){
                    List<MarketingStrategyBudgetDto> budgetShares = budgetMap.get(marketingStrategyItemDto.getStrategyItemCode());
                    marketingStrategyItemDto.setBudgetShares(budgetShares);
                    BigDecimal usableAmount = BigDecimal.ZERO;
                    if (!CollectionUtils.isEmpty(budgetShares)){
                        for (MarketingStrategyBudgetDto budgetShare : budgetShares) {
                            usableAmount = usableAmount.add(budgetShare.getUseAmount().subtract(Optional.ofNullable(budgetShare.getUsedAmount()).orElse(BigDecimal.ZERO)));
                        }
                    }
                    marketingStrategyItemDto.setUsableAmount(usableAmount);
                }
            }
        }
        return dtoList;
    }

    /**
     * 转换字段
     */
    public void convertMonthBudgetProperty(List<MarketingStrategyBudgetDto> list){
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        Map<String,String> feeSourceMap = Maps.newHashMap();
        if (null != dictDataVoService){
            try {
                List<DictDataVo> feeSourceList = dictDataVoService.findByDictTypeCode(MonthBudgetConstant.DICT_TPM_FEE_BELONG_US);
                if (!CollectionUtils.isEmpty(feeSourceList)) {
                    feeSourceMap = feeSourceList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue, (o, n) -> o));
                }
            } catch (Exception e) {
                log.error("费用归口数据字典查询失败：" + e.getMessage());
            }
        }
        for (MarketingStrategyBudgetDto vo : list) {
            //费用来源转换
            if (org.apache.commons.lang3.StringUtils.isNotEmpty(vo.getFeeBelongCode())){
                vo.setFeeSourceName(feeSourceMap.getOrDefault(vo.getFeeBelongCode(),vo.getFeeBelongCode()));
            }
        }
    }


    /**
     * 根据策略编码查询策略明细列表
     */
    public List<MarketingStrategyItemDto> findDtoListByStrategyCode(String strategyCode){
        if (StringUtils.isBlank(strategyCode)){
            return null;
        }
        List<MarketingStrategyItem> list = findListByStrategyCode(strategyCode);
        List<MarketingStrategyItemDto> dtoList = (List<MarketingStrategyItemDto>) nebulaToolkitService.copyCollectionByWhiteList(list, MarketingStrategyItem.class, MarketingStrategyItemDto.class, HashSet.class, ArrayList.class);
        fillDtoListProperties(dtoList);
        return dtoList;
    }

    /**
     * 根据策略编码查询策略明细列表
     */
    public List<MarketingStrategyItemDto> findDtoList(MarketingStrategyItemDto dto) {
        List<MarketingStrategyItem> list = findList(dto);
        List<MarketingStrategyItemDto> dtoList = (List<MarketingStrategyItemDto>) nebulaToolkitService.copyCollectionByWhiteList(list, MarketingStrategyItem.class, MarketingStrategyItemDto.class, HashSet.class, ArrayList.class);
        fillDtoListProperties(dtoList);
        return dtoList;
    }
    /**
     * 根据策略编码查询策略明细列表
     */
    public List<MarketingStrategyItem> findListByStrategyCode(String strategyCode) {
        if (StringUtils.isBlank(strategyCode)){
            return null;
        }

        MarketingStrategyItemDto dto = new MarketingStrategyItemDto();
        dto.setStrategyCode(strategyCode);
        return findList(dto);
    }

    /**
     * 根据查询条件筛选集合
     * @param dto 查询参数
     */
    public List<MarketingStrategyItem> findList(MarketingStrategyItemDto dto) {
        LambdaQueryWrapper<MarketingStrategyItem> wrapper = Wrappers.lambdaQuery(MarketingStrategyItem.class)
                .eq(StringUtils.isNotBlank(dto.getStrategyCode()), MarketingStrategyItem::getStrategyCode, dto.getStrategyCode())
                .eq(MarketingStrategyItem::getTenantCode, TenantUtils.getTenantCode())
                .eq(MarketingStrategyItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
        if (!CollectionUtils.isEmpty(dto.getStrategyCodeList())){
            wrapper.in(MarketingStrategyItem::getStrategyCode,dto.getStrategyCodeList());
        }
        if (!CollectionUtils.isEmpty(dto.getStrategyItemCodeList())){
            wrapper.in(MarketingStrategyItem::getStrategyItemCode,dto.getStrategyItemCodeList());
        }
        return this.list(wrapper);
    }

    public List<String> findStrategyCodeListByItemCodeList(List<String> itemCodeList){
        if (CollectionUtils.isEmpty(itemCodeList)){
            return Lists.newArrayList();
        }
        return this.list(Wrappers.lambdaQuery(MarketingStrategyItem.class)
                .in(MarketingStrategyItem::getStrategyItemCode, itemCodeList)
                .eq(MarketingStrategyItem::getTenantCode, TenantUtils.getTenantCode())
                .eq(MarketingStrategyItem::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .select(MarketingStrategyItem::getStrategyCode)
        ).stream().map(MarketingStrategyItem::getStrategyCode).distinct().collect(Collectors.toList());
    }

    /**
     * 通过id删除
     * @param ids id集合
     */
    public void deleteByIds(ArrayList<String> ids) {
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        LambdaUpdateWrapper<MarketingStrategyItem> updateWrapper = new UpdateWrapper<MarketingStrategyItem>().lambda();
        updateWrapper.set(MarketingStrategyItem::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(MarketingStrategyItem::getId, ids);
        updateWrapper.eq(MarketingStrategyItem::getTenantCode, TenantUtils.getTenantCode());
        this.update(updateWrapper);
    }

    public void deleteByStrategyCodes(List<String> strategyCodes) {
        if (CollectionUtils.isEmpty(strategyCodes)){
            return;
        }
        LambdaUpdateWrapper<MarketingStrategyItem> updateWrapper = new UpdateWrapper<MarketingStrategyItem>().lambda();
        updateWrapper.set(MarketingStrategyItem::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(MarketingStrategyItem::getStrategyCode, strategyCodes);
        updateWrapper.eq(MarketingStrategyItem::getTenantCode, TenantUtils.getTenantCode());
        this.update(updateWrapper);
    }

    /**
     * 根据策略编码列举关联的预算编码
     * @param strategyCodes 策略编码集合
     * @return 预算编码集合
     */
    public List<String> listBudgetCodeListByStrategyCodeList(List<String> strategyCodes) {
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(strategyCodes)) {
            return null;
        }
        List<MarketingStrategyItem> list = this.query()
                .in("plan_code", strategyCodes)
                .eq("del_flag", DelFlagStatusEnum.NORMAL.getCode())
                .isNotNull("month_budget_code")
                .select("distinct month_budget_code")
                .list();
        return list.stream().map(MarketingStrategyItem::getMonthBudgetCode).collect(Collectors.toList());
    }


    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoListProperties(List<MarketingStrategyItemVo> voList) {
        if (CollectionUtils.isEmpty(voList)){
            return;
        }
        for (MarketingStrategyItemVo item : voList) {
            this.fillVoProperties(item);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(MarketingStrategyItemVo item) {
        if (null == item){
            return;
        }
        ObjectConvertStringUtil.fillObjectStrProperties(item,MarketingStrategyItemVo.class);
        item.setActivityType(item.getActivityTypeCode());
        item.setActivityForm(item.getActivityFormCode());
        BigDecimal feeAmount = Optional.ofNullable(item.getFeeAmount()).orElse(BigDecimal.ZERO);
        BigDecimal usedAmount = Optional.ofNullable(item.getUsedAmount()).orElse(BigDecimal.ZERO);
        item.setUsedAmount(usedAmount);
        item.setUsableAmount(feeAmount.subtract(usedAmount));
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillDtoListProperties(List<? extends MarketingStrategyItemDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)){
            return;
        }
        for (MarketingStrategyItemDto item : dtoList) {
            this.fillDtoProperties(item);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillDtoProperties(MarketingStrategyItemDto item) {
        if (null == item){
            return;
        }
        ObjectConvertStringUtil.fillObjectStrProperties(item,MarketingStrategyItemDto.class);
        item.setActivityType(item.getActivityTypeCode());
        item.setActivityForm(item.getActivityFormCode());

//        MarketingStrategyItemScope scopeDto = new MarketingStrategyItemScope();
//        scopeDto.setScopeCode(item.getStrategyOrgCode());
//        scopeDto.setScopeName(item.getStrategyOrgName());
//        scopeDto.setSystemShareAmount(item.getFeeAmountStr());
//        scopeDto.setSureAmount(scopeDto.getSystemShareAmount());
//        scopeDto.setAdjustAmount("0");
//        scopeDto.setControlRatio(item.getControlRatio());
//        item.setScopeList(Lists.newArrayList(scopeDto));
    }

    public List<MarketingStrategyItemVo> findByCodes(List<String> itemCodes) {
        return this.baseMapper.findByCodes(itemCodes);
    }
    public Integer itemExportGetTotal(MarketingStrategyItemDto dto) {
        return this.baseMapper.itemExportGetTotal(dto);
    }

    public Page<MarketStrategyItemExportVo> itemExportGetDate(Page<MarketStrategyItemExportVo> page, MarketingStrategyItemDto dto) {
        return this.baseMapper.itemExportGetDate(page,dto);
    }

    public void updateCloseStatus(List<String> strategyItemCodeList) {
        if (CollectionUtils.isEmpty(strategyItemCodeList)){
            return;
        }
        LambdaUpdateWrapper<MarketingStrategyItem> updateWrapper = new UpdateWrapper<MarketingStrategyItem>().lambda();
        updateWrapper.set(MarketingStrategyItem::getIsClose, BooleanEnum.TRUE.getCapital());
        updateWrapper.in(MarketingStrategyItem::getStrategyItemCode, strategyItemCodeList);
        updateWrapper.eq(MarketingStrategyItem::getTenantCode, TenantUtils.getTenantCode());
        this.update(updateWrapper);

        //更新下策略头状态
        marketingStrategyBudgetRepository.updateHeadStatus(strategyItemCodeList);
    }
}

