package com.biz.crm.tpm.business.marketing.strategy.local.repository;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.core.toolkit.support.SFunction;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.util.NumberStringDealUtil;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategy;
import com.biz.crm.tpm.business.marketing.strategy.local.mapper.MarketingStrategyMapper;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;


/**
 * 营销策略(MarketingStrategy)表数据库访问层
 *
 * @author wanghaojia
 * @since 2022-11-05 16:06:03
 */
@Component
public class MarketingStrategyRepository extends ServiceImpl<MarketingStrategyMapper, MarketingStrategy> {

    @Autowired(required = false)
    private MarketingStrategyMapper marketingStrategyMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    public void deleteByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        LambdaUpdateWrapper<MarketingStrategy> updateWrapper = new UpdateWrapper<MarketingStrategy>().lambda();
        updateWrapper.set(MarketingStrategy::getDelFlag, DelFlagStatusEnum.DELETE.getCode());
        updateWrapper.in(MarketingStrategy::getId, ids);
        this.update(updateWrapper);
    }

    /**
     * 按条件查询营销策略
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     */
    public Page<MarketingStrategyVo> findByConditions(Pageable pageable, MarketingStrategyDto dto) {
        Page<MarketingStrategyVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<MarketingStrategyVo> pageList = this.marketingStrategyMapper.findByConditions(page, dto);
        fillVoListProperties(pageList.getRecords());
        return pageList;
    }

    /**
     * 根据查询条件筛选集合
     *
     * @param dto 查询参数
     * @param columns 要查询的列
     */
    public final List<MarketingStrategy> findList(MarketingStrategyDto dto) {
        return findList(dto,null);
    }
    /**
     * 根据查询条件筛选集合
     *
     * @param dto 查询参数
     * @param columns 要查询的列
     */
    @SafeVarargs
    public final List<MarketingStrategy> findList(MarketingStrategyDto dto, SFunction<MarketingStrategy, ?>... columns) {
        LambdaQueryWrapper<MarketingStrategy> wrapper = Wrappers.lambdaQuery(MarketingStrategy.class);
        wrapper.eq(StringUtils.isNotBlank(dto.getStrategyCode()), MarketingStrategy::getStrategyCode, dto.getStrategyCode());
        wrapper.eq(StringUtils.isNotBlank(dto.getBusinessFormatCode()), MarketingStrategy::getBusinessFormatCode, dto.getBusinessFormatCode());
        wrapper.eq(StringUtils.isNotBlank(dto.getBusinessUnitCode()), MarketingStrategy::getBusinessUnitCode, dto.getBusinessUnitCode());
        wrapper.eq(StringUtils.isNotBlank(dto.getAttrOrgCode()), MarketingStrategy::getAttrOrgCode, dto.getAttrOrgCode());
        wrapper.eq(StringUtils.isNotBlank(dto.getStrategyType()), MarketingStrategy::getStrategyType, dto.getStrategyType());
        wrapper.eq(StringUtils.isNotBlank(dto.getStrategyYear()), MarketingStrategy::getStrategyYear, dto.getStrategyYear());
        wrapper.eq(StringUtils.isNotBlank(dto.getCreateAccount()), MarketingStrategy::getCreateAccount, dto.getCreateAccount());
        wrapper.eq(StringUtils.isNotBlank(dto.getProcessStatus()), MarketingStrategy::getProcessStatus, dto.getProcessStatus());
        wrapper.eq(MarketingStrategy::getIsValidate, BooleanEnum.TRUE.getCapital());
        if (!CollectionUtils.isEmpty(dto.getStrategyCodeList())) {
            wrapper.in(MarketingStrategy::getStrategyCode, dto.getStrategyCodeList());
        }
        if (!CollectionUtils.isEmpty(dto.getProcessStatusList())) {
            wrapper.in(MarketingStrategy::getProcessStatus, dto.getProcessStatusList());
        }
        if (!CollectionUtils.isEmpty(dto.getStrategyCodeList())) {
            wrapper.in(MarketingStrategy::getStrategyCode, dto.getStrategyCodeList());
        }
        if (!CollectionUtils.isEmpty(dto.getAttrOrgCodeList())) {
            wrapper.in(MarketingStrategy::getAttrOrgCode, dto.getAttrOrgCodeList());
        }
        wrapper.eq(MarketingStrategy::getTenantCode, TenantUtils.getTenantCode())
                .eq(MarketingStrategy::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
        if (null != columns && columns.length > 0){
            wrapper.select(columns);
        }
        return this.list(wrapper);
    }

    /**
     * 根据查询条件筛选集合
     *
     * @param dto 查询参数
     */
    public List<String> findCodeList(MarketingStrategyDto dto) {
        return findList(dto,MarketingStrategy::getStrategyCode).stream().map(MarketingStrategy::getStrategyCode).collect(Collectors.toList());
    }

    public void updateProcessStatus(List<String> codeList, String processStatus) {
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }
        this.update(Wrappers.lambdaUpdate(MarketingStrategy.class)
                .set(MarketingStrategy::getProcessStatus, processStatus)
                .in(MarketingStrategy::getStrategyCode, codeList)
                .eq(MarketingStrategy::getTenantCode, TenantUtils.getTenantCode())
        );
    }

    public void updateProcessStatusAndProcessNo(List<String> codeList, String processStatus, String processNo) {
        if (CollectionUtils.isEmpty(codeList)) {
            return;
        }
        this.update(Wrappers.lambdaUpdate(MarketingStrategy.class)
                .set(MarketingStrategy::getProcessStatus, processStatus)
                .set(MarketingStrategy::getProcessNo, processNo)
                .in(MarketingStrategy::getStrategyCode, codeList)
                .eq(MarketingStrategy::getTenantCode, TenantUtils.getTenantCode())
        );
    }

    public MarketingStrategyVo getVoById(String id) {
        MarketingStrategy entity = this.getById(id);
        if (null == entity){
            return null;
        }
        MarketingStrategyVo vo = nebulaToolkitService.copyObjectByWhiteList(entity, MarketingStrategyVo.class, HashSet.class, ArrayList.class);
        fillVoProperties(vo);
        return vo;
    }



    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoListProperties(List<MarketingStrategyVo> voList) {
        if (CollectionUtils.isEmpty(voList)){
            return;
        }
        for (MarketingStrategyVo item : voList) {
            this.fillVoProperties(item);
        }
    }

    /**
     * 设置VO里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public void fillVoProperties(MarketingStrategyVo item) {
        if (null == item){
            return;
        }
        NumberStringDealUtil.setStringIfNotNull(item.getStrategySign(), item::setStrategySignStr);
        NumberStringDealUtil.setStringIfNotNull(item.getFeeAmount(), item::setFeeAmountStr);
        BigDecimal feeAmount = Optional.ofNullable(item.getFeeAmount()).orElse(BigDecimal.ZERO);
        BigDecimal usedAmount = Optional.ofNullable(item.getUsedAmount()).orElse(BigDecimal.ZERO);
        BigDecimal closeAmount = Optional.ofNullable(item.getCloseAmount()).orElse(BigDecimal.ZERO);
        item.setUsedAmount(usedAmount);
        item.setUsableAmount(feeAmount.subtract(usedAmount).subtract(closeAmount));
    }

    /**
     * 更新营销策略头上的费用金额
     * @param strategyCodeList 营销策略编码
     */
    public void updateFeeAmount(List<String> strategyCodeList) {
        marketingStrategyMapper.updateFeeAmount(strategyCodeList);
    }
}

