package com.biz.crm.tpm.business.marketing.strategy.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategy;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategyBudget;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyBudgetRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyItemRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyRepository;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyBudgetDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.OperateMarketingStrategyBudgetDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.service.MarketingStrategyBudgetSdkService;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyBudgetVo;
import com.biz.crm.tpm.business.month.budget.sdk.constant.BudgetLockConstant;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDetailDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetLockService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author wanghaojia
 * @date 2023/2/13 18:21
 */
@Service
public class MarketingStrategyBudgetSdkServiceImpl implements MarketingStrategyBudgetSdkService {

    @Autowired(required = false)
    private MarketingStrategyBudgetRepository marketingStrategyBudgetRepository;

    @Autowired(required = false)
    private MarketingStrategyItemRepository marketingStrategyItemRepository;

    @Autowired(required = false)
    private MarketingStrategyRepository marketingStrategyRepository;

    @Autowired(required = false)
    private MarketingStrategyBudgetSdkServiceImpl marketingStrategyBudgetSdkService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private MonthBudgetLockService monthBudgetLockService;


    @Override
    public List<MarketingStrategyBudgetDto> findListByConditions(MarketingStrategyBudgetDto dto) {
        return marketingStrategyBudgetRepository.findListByConditions(dto);
    }

    @Override
    public List<String> listBudgetCodeListByStrategyCodeList(List<String> strategyCodes) {
        return marketingStrategyItemRepository.listBudgetCodeListByStrategyCodeList(strategyCodes);
    }

    @Override
    public Page<MarketingStrategyBudgetVo> findByConditions(Pageable pageable, MarketingStrategyBudgetDto dto) {
        return marketingStrategyBudgetRepository.findByConditions(pageable,dto);
    }

    @Override
    public void operateAmount(List<OperateMarketingStrategyBudgetDto> operateList) {
        if (CollectionUtils.isEmpty(operateList)) {
            return;
        }
        Set<String> monthBudgetCodeSet = operateList.stream().map(OperateMarketingStrategyBudgetDto::getMonthBudgetCode).filter(Objects::nonNull).collect(Collectors.toSet());
        try {
            Validate.isTrue(monthBudgetLockService.lock(new ArrayList<>(monthBudgetCodeSet), TimeUnit.SECONDS, BudgetLockConstant.DEFAULT_LOCK_TIME), "预算加锁失败，请稍后重试");
            operateAmount(operateList, Maps.newHashMap());
        } finally {
            monthBudgetLockService.unLock(new ArrayList<>(monthBudgetCodeSet));
        }
    }

    @Override
    public void operateAmount(List<OperateMarketingStrategyBudgetDto> operateList, Map<String, BigDecimal> looseAmountMap) {
        operateList.forEach(dto -> {
            if (!redisLockService.isLock(BudgetLockConstant.MONTH_BUDGET_LOCK + dto.getMonthBudgetCode())) {
                throw new RuntimeException("预算操作失败，月度预算" + dto.getMonthBudgetCode() + "未加锁！");
            }
        });
        List<MarketingStrategyBudgetDto> budgetDtos = validateOperateBudget(operateList);
        Map<String, MarketingStrategyBudgetDto> budgetEntityMap = budgetDtos.stream().collect(Collectors.toMap(item -> item.getStrategyItemCode() + item.getMonthBudgetCode(), Function.identity()));
        List<String> strategyCodeList = budgetDtos.stream().map(MarketingStrategyBudgetDto::getStrategyCode).distinct().collect(Collectors.toList());
        MarketingStrategyDto marketingStrategyDto = new MarketingStrategyDto();
        marketingStrategyDto.setStrategyCodeList(strategyCodeList);
        List<MarketingStrategy> entityList = marketingStrategyRepository.findList(marketingStrategyDto);
        Map<String, MarketingStrategy> strategyMap = entityList.stream().collect(Collectors.toMap(MarketingStrategy::getStrategyCode, Function.identity()));

        List<MonthBudgetDetailDto> detailList = Lists.newArrayList();
        for (OperateMarketingStrategyBudgetDto operateMonthBudgetDto : operateList) {
            String budgetKey = operateMonthBudgetDto.getStrategyItemCode() + operateMonthBudgetDto.getMonthBudgetCode();
            MarketingStrategyBudgetDto monthBudgetEntity = budgetEntityMap.get(budgetKey);
            String operationType = operateMonthBudgetDto.getOperationType();
            BigDecimal operationAmount = operateMonthBudgetDto.getOperationAmount();
            BigDecimal looseAmount = looseAmountMap.getOrDefault(budgetKey,BigDecimal.ZERO);
//            BigDecimal usableAmount = monthBudgetEntity.getUseAmount().subtract(monthBudgetEntity.getUsedAmount()).add(looseAmount);
            MarketingStrategy marketingStrategy = strategyMap.get(monthBudgetEntity.getStrategyCode());
            BigDecimal strategyUsedAmount = Optional.ofNullable(marketingStrategy.getUsedAmount()).orElse(BigDecimal.ZERO);
            BigDecimal closeAmount = Optional.ofNullable(marketingStrategy.getCloseAmount()).orElse(BigDecimal.ZERO);
            BigDecimal usableAmount = Optional.ofNullable(marketingStrategy.getFeeAmount()).orElse(BigDecimal.ZERO)
                    .subtract(strategyUsedAmount).subtract(closeAmount).add(looseAmount);
            // 根据预算类型操作预算
            if (BudgetOperationTypeEnum.USE.getCode().equals(operationType)) {
                Validate.isTrue(usableAmount.compareTo(operationAmount) >= 0, "["+operateMonthBudgetDto.getBusinessCode()+"]使用金额["+operationAmount+"]大于当前策略可用金额" + usableAmount + "，请检查！");
                monthBudgetEntity.setUsedAmount(monthBudgetEntity.getUsedAmount().add(operationAmount));
                marketingStrategy.setUsedAmount(strategyUsedAmount.add(operationAmount));
            }else if (BudgetOperationTypeEnum.RETURN.getCode().equals(operationType)){
                monthBudgetEntity.setUsedAmount(monthBudgetEntity.getUsedAmount().subtract(operationAmount));
                marketingStrategy.setUsedAmount(strategyUsedAmount.subtract(operationAmount));
            }else {
                throw new RuntimeException("策略预算操作类型有误！");
            }
        }
        Boolean doSave = operateList.get(0).getDoSave();
        if (null == doSave || doSave){
            marketingStrategyBudgetSdkService.updateOperateBudget(budgetEntityMap.values(),detailList);
        }
    }

    public List<MarketingStrategyBudgetDto> validateOperateBudget(List<OperateMarketingStrategyBudgetDto> operateList) {
        for (OperateMarketingStrategyBudgetDto operateMonthBudgetDto : operateList) {
            Validate.notEmpty(operateMonthBudgetDto.getMonthBudgetCode(), "操作预算时，预算编码不能为空！");
            Validate.notNull(operateMonthBudgetDto.getOperationAmount(), "操作预算时，操作金额不能为空！");
            Validate.notEmpty(operateMonthBudgetDto.getOperationType(), "操作预算时，操作类型不能为空！");
            Validate.notEmpty(operateMonthBudgetDto.getStrategyItemCode(), "操作策略预算时，策略明细编码不能为空！");
        }

        List<String> operateCodeList = operateList.stream().map(item -> item.getStrategyItemCode()+item.getMonthBudgetCode()).distinct().collect(Collectors.toList());
        List<MarketingStrategyBudgetDto> budgetEntityList = marketingStrategyBudgetRepository.listBudgetInfoByOperateList(operateCodeList);
        if (budgetEntityList.size() < operateCodeList.size()){
            List<String> existsCodes = budgetEntityList.stream().map(item -> item.getStrategyItemCode()+item.getMonthBudgetCode()).collect(Collectors.toList());
            String notExistsJoinCodesStr = operateCodeList.stream().filter(item -> !existsCodes.contains(item)).collect(Collectors.joining(","));
            throw new RuntimeException("策略预算占用操作失败，策略明细[" + notExistsJoinCodesStr + "]查询失败，请检查策略是否启用或是否存在！！");
        }
        for (MarketingStrategyBudgetDto marketingStrategyBudgetDto : budgetEntityList) {
//            关联策略明细[策略明细编码]已被关闭，请重新选择可用策略明细。
            if (BooleanEnum.TRUE.getCapital().equals(marketingStrategyBudgetDto.getIsClose())){
                throw new RuntimeException("关联策略明细[" + marketingStrategyBudgetDto.getStrategyItemCode() + "]已被关闭，请重新选择可用策略明细");
            }

            if (null == marketingStrategyBudgetDto.getUseAmount()){
                marketingStrategyBudgetDto.setUseAmount(BigDecimal.ZERO);
            }
            if (null == marketingStrategyBudgetDto.getUsedAmount()){
                marketingStrategyBudgetDto.setUsedAmount(BigDecimal.ZERO);
            }
        }
        return budgetEntityList;
    }

    /**
     * 保存更新操作预算数据
     * @param budgetList 预算头数据
     * @param detailList 预算明细数据
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateOperateBudget(Collection<MarketingStrategyBudgetDto> budgetList, List<MonthBudgetDetailDto> detailList) {
        if (!CollectionUtils.isEmpty(budgetList)){
            Collection<MarketingStrategyBudget> marketingStrategyBudgets = nebulaToolkitService.copyCollectionByWhiteList(budgetList, MarketingStrategyBudgetDto.class, MarketingStrategyBudget.class, HashSet.class, ArrayList.class);
            this.marketingStrategyBudgetRepository.updateBatchById(marketingStrategyBudgets);
            //同步更新下策略表头的总金额
            List<String> strategyCodeList = budgetList.stream().map(MarketingStrategyBudgetDto::getStrategyCode).distinct().collect(Collectors.toList());
            marketingStrategyBudgetRepository.updateStrategyUsedAmount(strategyCodeList);
        }
    }


}
