package com.biz.crm.tpm.business.marketing.strategy.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.NumberStringDealUtil;
import com.biz.crm.mn.common.page.cache.service.MnPageCacheHelper;
import com.biz.crm.tpm.business.budget.item.sdk.enums.FeeBelongEnum;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyItemRepository;
import com.biz.crm.tpm.business.marketing.strategy.sdk.constant.MarketingStrategyConstant;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyBudgetDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyItemDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyItemVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 营销策略明细分页缓存处理
 * @author wanghaojia
 * @date 2022/12/16 15:25
 */
@Component
public class MarketingStrategyPageCacheHelper extends MnPageCacheHelper<MarketingStrategyItemVo, MarketingStrategyItemDto> {

    @Autowired(required = false)
    private MarketingStrategyItemRepository marketingStrategyItemRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Override
    public String getCacheKeyPrefix() {
        return MarketingStrategyConstant.MARKETING_STRATEGY_ITEM_CACHE_KEY_PREFIX;
    }

    @Override
    public Class<MarketingStrategyItemDto> getDtoClass() {
        return MarketingStrategyItemDto.class;
    }

    @Override
    public Class<MarketingStrategyItemVo> getVoClass() {
        return MarketingStrategyItemVo.class;
    }

    @Override
    public List<MarketingStrategyItemDto> findDtoListFromRepository(MarketingStrategyItemDto dto, String cacheKey) {
        if (StringUtils.isEmpty(dto.getStrategyCode())){
            return Lists.newArrayList();
        }

        //有策略编码，从数据库里面查出来放到缓存里面
        List<MarketingStrategyItemDto> dtoList = marketingStrategyItemRepository.findDtoAndAttachListByStrategyCode(dto.getStrategyCode());
        dtoList.sort(Comparator.comparing(MarketingStrategyItemDto::getStrategyItemCode));
        return dtoList;
    }

    @Override
    public List<MarketingStrategyItemDto> newItem(String cacheKey, List<MarketingStrategyItemDto> itemList) {
        MarketingStrategyItemDto newItem = new MarketingStrategyItemDto();
        newItem.setId(UUID.randomUUID().toString().replace("-", ""));
        newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        return Lists.newArrayList(newItem);
    }

    @Override
    public List<MarketingStrategyItemDto> copyItem(String cacheKey, List<MarketingStrategyItemDto> itemList) {
        List<MarketingStrategyItemDto> newItemList = (List<MarketingStrategyItemDto>) nebulaToolkitService.copyCollectionByBlankList(itemList, MarketingStrategyItemDto.class, MarketingStrategyItemDto.class, HashSet.class, ArrayList.class);
        for (MarketingStrategyItemDto newItem : newItemList) {
            newItem.setId(UUID.randomUUID().toString().replace("-", ""));
            newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            newItem.setStrategyItemCode(null);
        }
        return newItemList;
    }

    @Override
    public void updateItem(String cacheKey, List<MarketingStrategyItemDto> itemList) {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (null == requestAttributes) {
            return;
        }

        ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) requestAttributes;
        String businessUnitCode = servletRequestAttributes.getRequest().getParameter("businessUnitCode");
        if (BusinessUnitEnum.VERTICAL.getCode().equals(businessUnitCode)) {
            for (MarketingStrategyItemDto itemDto : itemList) {
                if (!CollectionUtils.isEmpty(itemDto.getBudgetShares())) {
                    //垂直选择预算逻辑处理
                    BigDecimal feeAmount = BigDecimal.ZERO;

                    Set<String> monthBudgetCodeSet = Sets.newHashSet();
                    Set<String> budgetItemCodeSet = Sets.newHashSet();
                    Set<String> budgetItemNameSet = Sets.newHashSet();

                    for (MarketingStrategyBudgetDto budgetDto : itemDto.getBudgetShares()) {
                        if (StringUtils.isEmpty(budgetDto.getFeeBelongCode())) {
                            //没有费用归口不处理
                            continue;
                        }
                        if (StringUtils.isNotEmpty(budgetDto.getUseAmountStr())) {
                            NumberStringDealUtil.validateNumberStrAndSet(budgetDto.getUseAmountStr(), "本次使用预算金额", false, budgetDto::setUseAmount, BigDecimal.class);
                            feeAmount = feeAmount.add(budgetDto.getUseAmount());
                        }

                        if (StringUtils.isNotEmpty(budgetDto.getMonthBudgetCode())) {
                            monthBudgetCodeSet.add(budgetDto.getMonthBudgetCode());
                        }
                        if (StringUtils.isNotEmpty(budgetDto.getBudgetItemCode())) {
                            budgetItemCodeSet.add(budgetDto.getBudgetItemCode());
                        }
                        if (StringUtils.isNotEmpty(budgetDto.getBudgetItemName())) {
                            budgetItemNameSet.add(budgetDto.getBudgetItemName());
                        }
                    }

                    itemDto.setFeeAmount(feeAmount);
                    itemDto.setFeeAmountStr(feeAmount.toString());
                    itemDto.setMonthBudgetCode(String.join(",", monthBudgetCodeSet));
                    itemDto.setBudgetItemCode(String.join(",", budgetItemCodeSet));
                    itemDto.setBudgetItemName(String.join(",", budgetItemNameSet));

                    //              费效比（费效比=费用投入/销量目标）
//              销量目标   salesGoalAmount
//              费效比      costEffectivenessRatio
                    BigDecimal costEffectivenessRatio = BigDecimal.ZERO;
                    if (StringUtils.isNotEmpty(itemDto.getFeeAmountStr())) {
                        feeAmount = new BigDecimal(itemDto.getFeeAmountStr());
                    }
                    if (null != itemDto.getSalesGoalAmount() && BigDecimal.ZERO.compareTo(itemDto.getSalesGoalAmount()) != 0) {
                        costEffectivenessRatio = feeAmount.divide(itemDto.getSalesGoalAmount(), 4, RoundingMode.HALF_UP);
                    }
                    itemDto.setCostEffectivenessRatio(costEffectivenessRatio);

                }
            }
        }
    }

    @Override
    public Object getDtoKey(MarketingStrategyItemDto marketingStrategyItemDto) {
        return marketingStrategyItemDto.getId();
    }

    @Override
    public String getCheckedStatus(MarketingStrategyItemDto marketingStrategyItemDto) {
        return marketingStrategyItemDto.getChecked();
    }
}
