package com.biz.crm.tpm.business.marketing.strategy.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.NumberStringDealUtil;
import com.biz.crm.tpm.business.marketing.strategy.local.dto.MarketingStrategyApproveDto;
import com.biz.crm.tpm.business.marketing.strategy.local.dto.MarketingStrategyApproveSubmitDto;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategy;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategyBudget;
import com.biz.crm.tpm.business.marketing.strategy.local.entity.MarketingStrategyItem;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyBudgetRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyItemRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.repository.MarketingStrategyRepository;
import com.biz.crm.tpm.business.marketing.strategy.local.service.MarketingStrategyBudgetService;
import com.biz.crm.tpm.business.marketing.strategy.local.service.MarketingStrategyItemService;
import com.biz.crm.tpm.business.marketing.strategy.local.service.MarketingStrategyService;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyApproveInfoVo;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyApproveProcessInfoVo;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyApproveVo;
import com.biz.crm.tpm.business.marketing.strategy.local.vo.MarketingStrategyBudgetSumVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.constant.MarketingStrategyConstant;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyBudgetDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyItemDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.MarketingStrategyOutDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.dto.log.MarketingStrategyLogEventDto;
import com.biz.crm.tpm.business.marketing.strategy.sdk.enums.StrategyStatusEnum;
import com.biz.crm.tpm.business.marketing.strategy.sdk.event.log.MarketingStrategyLogEventListener;
import com.biz.crm.tpm.business.marketing.strategy.sdk.service.MarketingStrategyLockSdkService;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyItemVo;
import com.biz.crm.tpm.business.marketing.strategy.sdk.vo.MarketingStrategyVo;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.tpm.business.year.budget.sdk.dto.YearBudgetDto;
import com.biz.crm.tpm.business.year.budget.sdk.service.YearBudgetSdkService;
import com.biz.crm.tpm.business.year.budget.sdk.vo.YearBudgetVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBatchBusinessService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantContextHolder;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 营销策略(MarketingStrategy)表服务实现类
 *
 * @author wanghaojia
 * @since 2022-11-05 16:06:03
 */
@Service("marketingStrategyService")
public class MarketingStrategyServiceImpl implements MarketingStrategyService {

    @Autowired(required = false)
    private MarketingStrategyRepository marketingStrategyRepository;


    @Autowired(required = false)
    private OrgVoService orgVoService;

    @Autowired(required = false)
    private MarketingStrategyItemRepository marketingStrategyItemRepository;

    @Autowired(required = false)
    private MarketingStrategyBudgetRepository marketingStrategyBudgetRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private MarketingStrategyItemService marketingStrategyItemService;


    @Autowired(required = false)
    private MarketingStrategyBudgetService marketingStrategyBudgetService;

    @Autowired(required = false)
    private ProcessBatchBusinessService processBatchBusinessService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    /**
     * 月度预算服务
     */
    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;
    /**
     * 月度预算服务
     */
    @Autowired(required = false)
    private YearBudgetSdkService yearBudgetService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private ProcessBusinessMappingService processBusinessMappingService;

    @Autowired(required = false)
    private MarketingStrategyLockSdkService marketingStrategyLockSdkService;

    /**
     * 策略保存新增方法
     * @param dto 单策略表头数据
     * @param itemCacheKey 明细缓存key
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveSingleMarketingStrategy(MarketingStrategyDto dto, String itemCacheKey) {
        boolean update = !StringUtils.isBlank(dto.getId());
        List<MarketingStrategyItemDto> itemCacheList = marketingStrategyItemService.findCacheList(itemCacheKey);
        if (BooleanEnum.TRUE.getCapital().equals(dto.getMultipleStrategyAdd())){
            //多策略新增
            saveMutiMarketingStrategy(dto,itemCacheList);
        }else{
            //单策略新增
            dto.setStrategySignStr(MarketingStrategyConstant.DEFAULT_STRATEGY_SIGN_STR);
            for (MarketingStrategyItemDto marketingStrategyItemDto : itemCacheList) {
                marketingStrategyItemDto.setStrategySignStr(MarketingStrategyConstant.DEFAULT_STRATEGY_SIGN_STR);
                marketingStrategyItemDto.setStrategyName(dto.getStrategyName());
                marketingStrategyItemDto.setStrategyTitle(dto.getStrategyTitle());
                marketingStrategyItemDto.setStrategyDesc(dto.getStrategyDesc());
                marketingStrategyItemDto.setIsCrossMonth(dto.getIsCrossMonth());
            }
            saveMarketingStrategy(dto,itemCacheList);
        }
        //数据保存完清理掉缓存
        marketingStrategyItemService.clearCache(itemCacheKey);


        //新增修改业务日志
        MarketingStrategyLogEventDto logEventDto = new MarketingStrategyLogEventDto();
        logEventDto.setNewest(dto);
        if (!update) {
            logEventDto.setOriginal(null);
            SerializableBiConsumer<MarketingStrategyLogEventListener, MarketingStrategyLogEventDto> onCreate =
                    MarketingStrategyLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, MarketingStrategyLogEventListener.class, onCreate);
        } else {
            MarketingStrategy oldEntity = marketingStrategyRepository.getById(dto.getId());
            MarketingStrategyDto oldDto = nebulaToolkitService.copyObjectByWhiteList(oldEntity, MarketingStrategyDto.class, HashSet.class, ArrayList.class);
            logEventDto.setOriginal(oldDto);
            SerializableBiConsumer<MarketingStrategyLogEventListener, MarketingStrategyLogEventDto> onUpdate =
                    MarketingStrategyLogEventListener::onUpdate;
            this.nebulaNetEventClient.publish(logEventDto, MarketingStrategyLogEventListener.class, onUpdate);
        }
    }

    /**
     * 多策略保存新增方法
     * @param itemCacheList 多策略行明细
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveMutiMarketingStrategy(MarketingStrategyDto dto,List<MarketingStrategyItemDto> itemCacheList) {
        boolean tempSave = Optional.ofNullable(dto.getTempSave()).orElse(false);
        if (tempSave) {
            marketingStrategyItemService.tempSaveCreateValidateList(dto,itemCacheList);
        }else {
            marketingStrategyItemService.createValidateList(dto,itemCacheList);
        }
        Map<String, List<MarketingStrategyItemDto>> strategyMap = itemCacheList.stream().collect(Collectors.groupingBy(MarketingStrategyItemDto::getStrategySignStr));
        //先循环校验一遍表头,保存的时候就不校验了，减少无效操作
        Map<String,MarketingStrategyDto> dtoMap = Maps.newHashMap();
        for (Map.Entry<String, List<MarketingStrategyItemDto>> entry : strategyMap.entrySet()) {
            MarketingStrategyDto thisDto = nebulaToolkitService.copyObjectByWhiteList(entry.getValue().get(0),MarketingStrategyDto.class,HashSet.class,ArrayList.class);
            thisDto.setBusinessUnitCode(dto.getBusinessUnitCode());
            thisDto.setBusinessFormatCode(dto.getBusinessFormatCode());
            thisDto.setStrategyType(dto.getStrategyType());
            thisDto.setId(null);
            if (!tempSave) {
                createValidate(thisDto,entry.getValue(),false);
            }
            dtoMap.put(entry.getKey(),thisDto);
        }
        for (Map.Entry<String, List<MarketingStrategyItemDto>> entry : strategyMap.entrySet()) {
            saveMarketingStrategy(dtoMap.get(entry.getKey()),entry.getValue(),false);
        }
    }

    /**
     * 查询可以选择的月度预算
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @Override
    public Page<MonthBudgetVo> findMonthBudgetByConditions(Pageable pageable, MonthBudgetDto dto) {
        Validate.notBlank(dto.getBusinessFormatCode(),"业态不能为空");
        Page<MonthBudgetVo> page = monthBudgetService.findByConditionsNoFilter(pageable, dto);
        updateMonthBudgetProperties(page.getRecords());
        return page;
    }

    /**
     * 查询可以选择的年度预算
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @Override
    public Page<YearBudgetVo> findYearBudgetByConditions(Pageable pageable, YearBudgetDto dto){
        Validate.notBlank(dto.getBusinessFormatCode(),"业态不能为空");
        Page<YearBudgetVo> page = yearBudgetService.findByConditions(pageable, dto);
        updateYearBudgetProperties(page.getRecords());
        return page;
    }

    /**
     * 更新预算属性
     */
    private void updateMonthBudgetProperties(List<MonthBudgetVo> list){
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        for (MonthBudgetVo monthBudgetVo : list) {
            if (null == monthBudgetVo.getAfterFreezeAmount()){
                monthBudgetVo.setInitResolveAmount(BigDecimal.ZERO);
            }
            if (null == monthBudgetVo.getUsedStrategyAmount()){
                monthBudgetVo.setUsedStrategyAmount(BigDecimal.ZERO);
            }
            monthBudgetVo.setUsableStrategyAmount(monthBudgetVo.getAfterFreezeAmount().subtract(monthBudgetVo.getUsedStrategyAmount()));
        }
    }

    /**
     * 更新预算属性
     */
    private void updateYearBudgetProperties(List<YearBudgetVo> list){
        if (CollectionUtils.isEmpty(list)){
            return;
        }
        for (YearBudgetVo yearBudgetVo : list) {
            if (null == yearBudgetVo.getBudgetTotalAmount()){
                yearBudgetVo.setBudgetTotalAmount(BigDecimal.ZERO);
            }
            if (null == yearBudgetVo.getUsedStrategyAmount()){
                yearBudgetVo.setUsedStrategyAmount(BigDecimal.ZERO);
            }
            yearBudgetVo.setUsableStrategyAmount(yearBudgetVo.getBudgetTotalAmount().subtract(yearBudgetVo.getUsedStrategyAmount()));
        }
    }

    /**
     * 单策略新增保存方法-默认走校验逻辑
     * @param dto 策略头数据
     * @param itemDtoList 策略明细数据
     */
    @Transactional(rollbackFor = Exception.class)
    public void saveMarketingStrategy(MarketingStrategyDto dto,List<MarketingStrategyItemDto> itemDtoList){
        saveMarketingStrategy(dto,itemDtoList,true);
    }

    /**
     * 单策略新增保存方法
     * @param dto 策略头数据
     * @param itemDtoList 策略明细数据
     * @param createValidate 是否走校验逻辑
     */
    @Transactional(rollbackFor = Exception.class)
    public void saveMarketingStrategy(MarketingStrategyDto dto,List<MarketingStrategyItemDto> itemDtoList,boolean createValidate){
        saveMarketingStrategy(dto, itemDtoList, createValidate, true);
    }

    /**
     * 单策略新增保存方法
     * @param dto 策略头数据
     * @param itemDtoList 策略明细数据
     * @param createValidate 是否走校验逻辑
     */
    @Transactional(rollbackFor = Exception.class)
    public void saveMarketingStrategy(MarketingStrategyDto dto,List<MarketingStrategyItemDto> itemDtoList,boolean createValidate,boolean createValidateItem){
        boolean tempSave = Optional.ofNullable(dto.getTempSave()).orElse(false);
        if (createValidate && !tempSave){
            createValidate(dto,itemDtoList,createValidateItem);
        }
        MarketingStrategy entity = null;
        boolean update = !StringUtils.isBlank(dto.getId());
        if (!update){
            entity = nebulaToolkitService.copyObjectByWhiteList(dto,MarketingStrategy.class,HashSet.class,ArrayList.class);
            // redis生成营销策略编码，编码规则为MS+年月日+5位顺序数。每天都从00001开始
//            String ruleCode = StringUtils.join(MarketingStrategyConstant.MARKETING_STRATEGY_RULE_CODE_PRE, DateFormatUtils.format(new Date(), "yyyyMMdd"));
            String code = this.generateCodeService.generateCode(MarketingStrategyConstant.MARKETING_STRATEGY_RULE_CODE_PRE, 1, 6, 2, TimeUnit.DAYS).get(0);
            entity.setStrategyCode(code);
            entity.setProcessStatus(ProcessStatusEnum.PREPARE.getKey());
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            entity.setTenantCode(TenantContextHolder.getTenantInfo().getTenantCode());
            entity.setIsValidate(!tempSave ? BooleanEnum.TRUE.getCapital() : BooleanEnum.FALSE.getCapital());
            entity.setStrategyStatus(StrategyStatusEnum.normal.getCode());
            marketingStrategyRepository.save(entity);
            dto.setId(entity.getId());
        }else{
            MarketingStrategy oldEntity = marketingStrategyRepository.getById(dto.getId());
            if (!ProcessStatusEnum.PREPARE.getKey().equals(oldEntity.getProcessStatus()) &&
                    !ProcessStatusEnum.REJECT.getKey().equals(oldEntity.getProcessStatus()) &&
                    !ProcessStatusEnum.RECOVER.getKey().equals(oldEntity.getProcessStatus())) {
                throw new RuntimeException("只能编辑待提交、驳回、追回状态的数据！");
            }
            //不允许修改的属性
            dto.setStrategyCode(oldEntity.getStrategyCode());
            dto.setStrategySign(oldEntity.getStrategySign());
            dto.setTenantCode(oldEntity.getTenantCode());
            dto.setProcessStatus(ProcessStatusEnum.PREPARE.getKey());
            dto.setIsValidate(!tempSave ? BooleanEnum.TRUE.getCapital() : BooleanEnum.FALSE.getCapital());
            entity = nebulaToolkitService.copyObjectByWhiteList(dto,MarketingStrategy.class,HashSet.class,ArrayList.class);
            marketingStrategyRepository.updateById(entity);
        }
        marketingStrategyItemService.saveMarketingStrategyItemList(entity,update,itemDtoList,false,tempSave);
        marketingStrategyBudgetService.saveMarketingStrategyBudgetList(entity,update,itemDtoList);
    }


    /**
     * 策略新增保存逻辑
     * @param dto 策略数据
     * @param itemCacheList 策略行数据
     */
    private void createValidate(MarketingStrategyDto dto,List<MarketingStrategyItemDto> itemCacheList,boolean validateItem) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        Validate.notBlank(dto.getStrategySignStr(),"新增时，策略明细策略标识不能为空");
        try {
            int strategySign = Integer.parseInt(dto.getStrategySignStr().trim());
            dto.setStrategySign(strategySign);
        }catch (NumberFormatException e){
            throw new RuntimeException("新增时，策略标识["+dto.getStrategySignStr()+"]格式有误,只能是整数");
        }
        Validate.notBlank(dto.getBusinessFormatCode(), "新增时，业态不能为空！");
        Validate.notBlank(dto.getStrategyName(), "新增时，策略名称不能为空！");
        Validate.notBlank(dto.getStrategyType(), "新增时，策略类型不能为空！");
        Validate.notBlank(dto.getAttrOrgCode(), "新增时，归属部门不能为空！");
        Validate.notBlank(dto.getStrategyTitle(), "新增时，策略主题不能为空！");
        dto.setStrategyStatus(StrategyStatusEnum.normal.getCode());
//        Validate.notBlank(dto.getStrategyDesc(), "新增时，策略描述不能为空！");
        if (validateItem){
            Validate.isTrue(!CollectionUtils.isEmpty(itemCacheList),"新增时，策略内容不能为空");
            marketingStrategyItemService.createValidateList(dto,itemCacheList);
        }

        if (!CollectionUtils.isEmpty(itemCacheList)){
            BigDecimal feeAmount = itemCacheList.stream().map(MarketingStrategyItemDto::getFeeAmount).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
            dto.setFeeAmount(feeAmount);

            String strategyYear = itemCacheList.get(0).getStrategyYearMonth().substring(0,4);
            dto.setStrategyYear(strategyYear);
            String beginStrategyYearMonth = itemCacheList.stream().map(MarketingStrategyItemDto::getStrategyYearMonth).min(Comparator.naturalOrder()).orElse(null);
            dto.setBeginStrategyYearMonth(beginStrategyYearMonth);
            String endStrategyYearMonth = itemCacheList.stream().map(MarketingStrategyItemDto::getStrategyYearMonth).max(Comparator.naturalOrder()).orElse(null);
            dto.setEndStrategyYearMonth(endStrategyYearMonth);
        }

        BigDecimal budgetFeeAmount = BigDecimal.ZERO;
        for (MarketingStrategyItemDto item : itemCacheList) {
            //业务单元不是垂直，清空掉明细上的预算信息，按明细上的预算信息重新保存
            if (!BusinessUnitEnum.VERTICAL.getCode().equals(dto.getBusinessUnitCode())){
                //其他业务单元重新组装预算信息
                MarketingStrategyBudgetDto budgetDto = new MarketingStrategyBudgetDto();
                budgetDto.setMonthBudgetCode(item.getMonthBudgetCode());
                budgetDto.setBudgetItemCode(item.getBudgetItemCode());
                budgetDto.setBudgetItemName(item.getBudgetItemName());
                budgetDto.setFeeBelongCode(item.getFeeBelongCode());
                budgetDto.setUseAmountStr(item.getFeeAmountStr());
                budgetDto.setUseAmount(item.getFeeAmount());
                budgetDto.setBusinessUnitCode(dto.getBusinessUnitCode());
                item.setBudgetShares(Lists.newArrayList(budgetDto));
            }else {
                for (MarketingStrategyBudgetDto budgetShare : item.getBudgetShares()) {
                    budgetShare.setBusinessUnitCode(dto.getBusinessUnitCode());
                    NumberStringDealUtil.validateNumberStrAndSet(budgetShare.getUseAmountStr(), "新增时，策略明细费用金额", false,budgetShare::setUseAmount, BigDecimal.class);
                    if (null != budgetShare.getUseAmount()){
                        budgetFeeAmount = budgetFeeAmount.add(budgetShare.getUseAmount());
                    }
                }
            }
        }

        if (BusinessUnitEnum.VERTICAL.getCode().equals(dto.getBusinessUnitCode())) {
            //垂直多选预算，校验预算金额之和要等于行上的费用金额
            BigDecimal feeAmount = Optional.ofNullable(dto.getFeeAmount()).orElse(BigDecimal.ZERO);
            if (feeAmount.compareTo(budgetFeeAmount) != 0){
                throw new RuntimeException("费用金额["+feeAmount+"]不等于实际预算使用金额["+budgetFeeAmount+"]");
            }
        }

        //批量校验预算信息
        List<MarketingStrategyBudgetDto> budgetDtos = Lists.newArrayList();
        for (MarketingStrategyItemDto itemDto : itemCacheList) {
            if (CollectionUtils.isEmpty(itemDto.getBudgetShares())){
                continue;
            }
            budgetDtos.addAll(itemDto.getBudgetShares());
        }
        marketingStrategyBudgetService.useMonthBudget(budgetDtos,false);
    }

    /**
     * 删除策略
     * @param ids 策略ids
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteMarketingStrategy(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)){
            return;
        }
        List<MarketingStrategy> marketingStrategies = marketingStrategyRepository.listByIds(ids);
        for (MarketingStrategy marketingStrategy : marketingStrategies) {
            Validate.isTrue(ProcessStatusEnum.PREPARE.getKey().equals(marketingStrategy.getProcessStatus()) ||
                    ProcessStatusEnum.REJECT.getKey().equals(marketingStrategy.getProcessStatus()) ||
                                    ProcessStatusEnum.RECOVER.getKey().equals(marketingStrategy.getProcessStatus()),
                    "营销策略["+marketingStrategy.getStrategyCode()+"]不是待提交、驳回或追回状态，不能删除！");
        }
        marketingStrategyRepository.deleteByIds(ids);
        List<String> strategyCodes = marketingStrategies.stream().map(MarketingStrategy::getStrategyCode).collect(Collectors.toList());
        marketingStrategyItemRepository.deleteByStrategyCodes(strategyCodes);
        marketingStrategyBudgetService.deleteByStrategyCodes(strategyCodes);
        
        //删除业务日志
        Collection<MarketingStrategyDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(marketingStrategies,
                MarketingStrategy.class, MarketingStrategyDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<MarketingStrategyLogEventListener, MarketingStrategyLogEventDto> onDelete =
                MarketingStrategyLogEventListener::onDelete;
        for (MarketingStrategyDto dto : dtoList) {
            MarketingStrategyLogEventDto logEventDto = new MarketingStrategyLogEventDto();
            logEventDto.setOriginal(dto);
            this.nebulaNetEventClient.publish(logEventDto, MarketingStrategyLogEventListener.class, onDelete);
        }
    }

    /**
     * 按条件查询营销策略
     * @param pageable 分页参数
     * @param dto 查询参数
     */
    @Override
    public Page<MarketingStrategyVo> findByConditions(Pageable pageable, MarketingStrategyDto dto) {
        return marketingStrategyRepository.findByConditions(pageable, dto);
    }

    /**
     * 通过id获取营销策略数据
     */
    @Override
    public MarketingStrategyVo findById(String id) {
        if (StringUtils.isEmpty(id)){
            return null;
        }
        return marketingStrategyRepository.getVoById(id);
    }

    /**
     * 获取流程审批信息
     */
    @Override
    public List<MarketingStrategyApproveInfoVo> findApproveInfoList(MarketingStrategyApproveDto dto) {
        //找到所有待提交、驳回、追回的流程数据提交
        MarketingStrategyDto searchDto = nebulaToolkitService.copyObjectByWhiteList(dto,MarketingStrategyDto.class,HashSet.class,ArrayList.class);
        List<String> processStatusList = Lists.newArrayList();
        processStatusList.add(ProcessStatusEnum.PREPARE.getKey());
        processStatusList.add(ProcessStatusEnum.REJECT.getKey());
        processStatusList.add(ProcessStatusEnum.RECOVER.getKey());
        searchDto.setProcessStatusList(processStatusList);
        if (!StringUtils.isEmpty(dto.getAttrOrgCode())){
            List<String> attrOrgCodeList = Arrays.asList(dto.getAttrOrgCode().split(","));
            List<OrgVo> childOrgList = orgVoService.findAllChildrenByOrgCodes(attrOrgCodeList);
            searchDto.setAttrOrgCode(null);
            if (!CollectionUtils.isEmpty(childOrgList)){
                searchDto.setAttrOrgCodeList(childOrgList.stream().map(OrgVo::getOrgCode).collect(Collectors.toList()));
            }else{
                searchDto.setAttrOrgCodeList(attrOrgCodeList);
            }
        }else {
            throw new RuntimeException("归属部门不能为空");
        }
        List<String> strategyCodeList = marketingStrategyRepository.findCodeList(searchDto);

        return findApproveInfoList(strategyCodeList);
    }

    /**
     * 获取流程审批信息
     */
    @Override
    public List<MarketingStrategyApproveInfoVo> findApproveInfoList(List<String> strategyCodeList) {
        if (CollectionUtils.isEmpty(strategyCodeList)) {
            return Lists.newArrayList();
        }
        MarketingStrategyItemDto marketingStrategyItemDto = new MarketingStrategyItemDto();
        marketingStrategyItemDto.setStrategyCodeList(strategyCodeList);
        List<MarketingStrategyItem> list = marketingStrategyItemRepository.findList(marketingStrategyItemDto);
        return findApproveInfoListByItemList(list);
    }
    /**
     * 获取流程审批信息
     */
    public List<MarketingStrategyApproveInfoVo> findApproveInfoListByItemList(List<MarketingStrategyItem> list) {
        //先按预算项目+策略分组
        Map<String, List<MarketingStrategyItem>> firstGroupMap = list.stream().collect(Collectors.groupingBy(marketingStrategyItem -> {
            return marketingStrategyItem.getBudgetItemCode() + marketingStrategyItem.getStrategyCode();
        }));

        List<MarketingStrategyApproveInfoVo> result = Lists.newArrayList();
        for (Map.Entry<String, List<MarketingStrategyItem>> entry : firstGroupMap.entrySet()) {
            MarketingStrategyApproveInfoVo headVo = new MarketingStrategyApproveInfoVo();
            headVo.setStrategyOrgName("策略内容");
            result.add(headVo);

            List<MarketingStrategyItem> firstGroupList = entry.getValue();
            //再按区域分组
            Map<String, List<MarketingStrategyItem>> secondGroupMap = firstGroupList.stream().collect(Collectors.groupingBy(item -> Optional.ofNullable(item.getStrategyOrgCode()).orElse("")));
            List<MarketingStrategyApproveInfoVo> resultGroupList = Lists.newArrayList();

            for (Map.Entry<String, List<MarketingStrategyItem>> secondEntry : secondGroupMap.entrySet()) {
                List<MarketingStrategyItem> secondGroupList = secondEntry.getValue();
                MarketingStrategyItem tempItem = secondGroupList.get(0);
                MarketingStrategyApproveInfoVo approveInfoVo = new MarketingStrategyApproveInfoVo();
                approveInfoVo.setBudgetItemCode(tempItem.getBudgetItemCode());
                approveInfoVo.setBudgetItemName(tempItem.getBudgetItemName());
                approveInfoVo.setStrategyCode(tempItem.getStrategyCode());
                approveInfoVo.setStrategyOrgName(tempItem.getStrategyOrgName());
                approveInfoVo.setFirstChannelCode(tempItem.getFirstChannelCode());
                approveInfoVo.setFirstChannelName(tempItem.getFirstChannelName());
                approveInfoVo.setActivityFormCode(tempItem.getActivityFormCode());
                approveInfoVo.setActivityFormName(tempItem.getActivityFormName());
                approveInfoVo.setActivityTypeCode(tempItem.getActivityTypeCode());
                approveInfoVo.setActivityTypeName(tempItem.getActivityTypeName());
                initApproveInfoMonthZero(approveInfoVo);
                for (MarketingStrategyItem marketingStrategyItem : secondGroupList) {
                    BigDecimal amount = Optional.ofNullable(marketingStrategyItem.getFeeAmount()).orElse(BigDecimal.ZERO);
                    int month = Integer.parseInt(marketingStrategyItem.getStrategyYearMonth().substring(5,7));
                    //分摊到每个月上
                    switch (month){
                        case 1:headVo.setJanuaryStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setJanuary(approveInfoVo.getJanuary().add(amount));break;
                        case 2:headVo.setFebruaryStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setFebruary(approveInfoVo.getFebruary().add(amount));break;
                        case 3:headVo.setMarchStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setMarch(approveInfoVo.getMarch().add(amount));break;
                        case 4:headVo.setAprilStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setApril(approveInfoVo.getApril().add(amount));break;
                        case 5:headVo.setMayStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setMay(approveInfoVo.getMay().add(amount));break;
                        case 6:headVo.setJuneStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setJune(approveInfoVo.getJune().add(amount));break;
                        case 7:headVo.setJulyStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setJuly(approveInfoVo.getJuly().add(amount));break;
                        case 8:headVo.setAugustStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setAugust(approveInfoVo.getAugust().add(amount));break;
                        case 9:headVo.setSeptemberStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setSeptember(approveInfoVo.getSeptember().add(amount));break;
                        case 10:headVo.setOctoberStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setOctober(approveInfoVo.getOctober().add(amount));break;
                        case 11:headVo.setNovemberStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setNovember(approveInfoVo.getNovember().add(amount));break;
                        case 12:headVo.setDecemberStr(marketingStrategyItem.getStrategyName()); approveInfoVo.setDecember(approveInfoVo.getDecember().add(amount));break;
                    }
                }
                resultGroupList.add(approveInfoVo);
            }
            result.addAll(resultGroupList);

            MarketingStrategyApproveInfoVo lastVo = new MarketingStrategyApproveInfoVo();
            lastVo.setStrategyOrgName("合计");
            initApproveInfoMonthZero(lastVo);
            for (MarketingStrategyApproveInfoVo approveInfoVo : resultGroupList) {
                lastVo.setJanuary(lastVo.getJanuary().add(approveInfoVo.getJanuary()));
                lastVo.setFebruary(lastVo.getFebruary().add(approveInfoVo.getFebruary()));
                lastVo.setMarch(lastVo.getMarch().add(approveInfoVo.getMarch()));
                lastVo.setApril(lastVo.getApril().add(approveInfoVo.getApril()));
                lastVo.setMay(lastVo.getMay().add(approveInfoVo.getMay()));
                lastVo.setJune(lastVo.getJune().add(approveInfoVo.getJune()));
                lastVo.setJuly(lastVo.getJuly().add(approveInfoVo.getJuly()));
                lastVo.setAugust(lastVo.getAugust().add(approveInfoVo.getAugust()));
                lastVo.setSeptember(lastVo.getSeptember().add(approveInfoVo.getSeptember()));
                lastVo.setOctober(lastVo.getOctober().add(approveInfoVo.getOctober()));
                lastVo.setNovember(lastVo.getNovember().add(approveInfoVo.getNovember()));
                lastVo.setDecember(lastVo.getDecember().add(approveInfoVo.getDecember()));
            }
            result.add(lastVo);
        }
        //获取所有明细
        return result;
    }

    /**
     * 初始化提交审批VO里面的月份金额为0
     * @param vo
     */
    private void initApproveInfoMonthZero(MarketingStrategyApproveInfoVo vo){
        vo.setJanuary(BigDecimal.ZERO);
        vo.setFebruary(BigDecimal.ZERO);
        vo.setMarch(BigDecimal.ZERO);
        vo.setApril(BigDecimal.ZERO);
        vo.setMay(BigDecimal.ZERO);
        vo.setJune(BigDecimal.ZERO);
        vo.setJuly(BigDecimal.ZERO);
        vo.setAugust(BigDecimal.ZERO);
        vo.setSeptember(BigDecimal.ZERO);
        vo.setOctober(BigDecimal.ZERO);
        vo.setNovember(BigDecimal.ZERO);
        vo.setDecember(BigDecimal.ZERO);
    }

    /**
     * 获取待审批的营销策略数据
     * @param dto 查询条件
     * @return 待审批的营销策略数据
     */
    @Override
    public List<String> findToApprovalCodeList(MarketingStrategyApproveDto dto) {
        Validate.notBlank(dto.getStrategyType(),"策略类型不能为空");
        Validate.notBlank(dto.getAttrOrgCode(),"归属部门不能为空");
        Validate.notBlank(dto.getBusinessFormatCode(),"业态不能为空");
        Validate.notBlank(dto.getBusinessUnitCode(),"业务单元不能为空");
        Validate.notBlank(dto.getStrategyYear(),"年度不能为空");

        //找到所有待提交、驳回、追回的流程数据提交
        MarketingStrategyDto searchDto = nebulaToolkitService.copyObjectByWhiteList(dto,MarketingStrategyDto.class,HashSet.class,ArrayList.class);
        List<String> processStatusList = Lists.newArrayList();
        processStatusList.add(ProcessStatusEnum.PREPARE.getKey());
        processStatusList.add(ProcessStatusEnum.REJECT.getKey());
        processStatusList.add(ProcessStatusEnum.RECOVER.getKey());
        searchDto.setProcessStatusList(processStatusList);
        return marketingStrategyRepository.findCodeList(searchDto);
    }

    /**
     * 更新流程状态
     * @param codeList 要更新的编码
     * @param processStatus 流程状态
     */
    @Override
    public void updateProcessStatus(List<String> codeList, String processStatus) {
        marketingStrategyRepository.updateProcessStatus(codeList,processStatus);
    }

    /**
     * 提交流程
     * @param toApprovalCodeList 待提交的编码
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submitApproval(List<String> toApprovalCodeList, MarketingStrategyApproveSubmitDto dto) {
        //占用预算
        MarketingStrategyItemDto searchItemDto = new MarketingStrategyItemDto();
        searchItemDto.setStrategyCodeList(toApprovalCodeList);
        boolean lockSuccess = marketingStrategyLockSdkService.lock(toApprovalCodeList, TimeUnit.MINUTES, 3);
        Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
        try {
            marketingStrategyBudgetService.useMonthBudgetByCodeList(toApprovalCodeList);
            ProcessBusinessDto processBusinessDto = dto.getProcessBusiness();
            processBusinessDto.setBusinessNoList(toApprovalCodeList);
            processBusinessDto.setBusinessCode(MarketingStrategyConstant.PROCESS_NAME_MARKETING_STRATEGY);
            String businessNo = dto.getStrategyYear() + "-" + dto.getStrategyType() + "-" + dto.getBusinessFormatCode() + "-" + dto.getBusinessUnitCode() + "-" + dto.getAttrOrgCode() + System.currentTimeMillis();
            processBusinessDto.setBusinessNo(businessNo);
            ProcessBusinessVo processBusinessVo = processBatchBusinessService.processStart(processBusinessDto);
            //提交成功
            marketingStrategyRepository.updateProcessStatusAndProcessNo(toApprovalCodeList, ProcessStatusEnum.COMMIT.getKey(), processBusinessVo.getProcessNo());

        } finally {
            marketingStrategyLockSdkService.unlock(toApprovalCodeList);
        }

    }

    /**
     * 营销策略审批通过
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void processPass(ProcessStatusDto dto) {
        marketingStrategyRepository.updateProcessStatus(dto.getBusinessNoList(),dto.getProcessStatus());
    }

    /**
     * 审批驳回|流程追回
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void processRejectAndRecover(ProcessStatusDto dto) {
        marketingStrategyBudgetService.returnMonthBudgetByCodeList(dto.getBusinessNoList());
        marketingStrategyRepository.updateProcessStatus(dto.getBusinessNoList(),dto.getProcessStatus());
    }
    /**
     * 根据流程编码获取营销策略审批数据
     * @param processNo 流程编码
     */
    @Override
    public MarketingStrategyApproveProcessInfoVo findApproveProcessInfo(String processNo) {
        ProcessBusinessMappingDto processBusinessMappingDto = new ProcessBusinessMappingDto();
        processBusinessMappingDto.setProcessNo(processNo);
        List<ProcessBusinessMappingVo> processBusinessMappingVoList = this.processBusinessMappingService.findMultiByByConditions(processBusinessMappingDto);
        if (CollectionUtils.isEmpty(processBusinessMappingVoList)){
            throw new RuntimeException("流程数据有误！");
        }
        List<String> businessNoList = processBusinessMappingVoList.stream().map(ProcessBusinessMappingVo::getBusinessNo).collect(Collectors.toList());

        //找到所有待提交、驳回、追回的流程数据提交
        MarketingStrategyItemDto marketingStrategyItemDto = new MarketingStrategyItemDto();
        marketingStrategyItemDto.setStrategyCodeList(businessNoList);
        List<MarketingStrategyItem> list = marketingStrategyItemRepository.findList(marketingStrategyItemDto);

        MarketingStrategyApproveProcessInfoVo processInfoVo = new MarketingStrategyApproveProcessInfoVo();

        MarketingStrategyDto marketingStrategyDto = new MarketingStrategyDto();
        marketingStrategyDto.setStrategyCode(businessNoList.get(0));
        List<MarketingStrategy> entityList = marketingStrategyRepository.findList(marketingStrategyDto);
        if (CollectionUtils.isEmpty(entityList)){
            throw new RuntimeException("策略信息有误！");
        }
        //设置下审批信息表头
        MarketingStrategy marketingStrategy = entityList.get(0);
        processInfoVo.setApproveHead( new MarketingStrategyApproveVo(){{
            this.setStrategyType(marketingStrategy.getStrategyType());
            this.setStrategyYear(marketingStrategy.getStrategyYear());
            this.setBusinessFormatCode(marketingStrategy.getBusinessFormatCode());
            this.setBusinessUnitCode(marketingStrategy.getBusinessUnitCode());
            this.setAttrOrgCode(marketingStrategy.getAttrOrgCode());
            this.setAttrOrgName(marketingStrategy.getAttrOrgName());
        }});

        List<MarketingStrategyApproveInfoVo> infoList = findApproveInfoListByItemList(list);
        processInfoVo.setInfoList(infoList);
        return processInfoVo;
    }

    @Override
    public Page<MarketingStrategyItemVo> findApproveProcessItemPage(Pageable pageable, MarketingStrategyApproveDto dto) {
        List<String> strategyCodeList = findToApprovalCodeList(dto);
        if (CollectionUtils.isEmpty(strategyCodeList)){
            throw new RuntimeException("未查询到待提交数据");
        }
        MarketingStrategyItemDto itemDto = new MarketingStrategyItemDto();
        itemDto.setStrategyCodeList(strategyCodeList);
        return marketingStrategyItemService.findByConditions(pageable,itemDto);
    }

    @Override
    public Page<MarketingStrategyItemVo> findApproveProcessItemPage(Pageable pageable, MarketingStrategyDto dto) {
        ProcessBusinessMappingDto processBusinessMappingDto = new ProcessBusinessMappingDto();
        processBusinessMappingDto.setProcessNo(dto.getProcessNo());
        List<ProcessBusinessMappingVo> processBusinessMappingVoList = this.processBusinessMappingService.findMultiByByConditions(processBusinessMappingDto);
        if (CollectionUtils.isEmpty(processBusinessMappingVoList)){
            throw new RuntimeException("流程数据有误！");
        }
        List<String> businessNoList = processBusinessMappingVoList.stream().map(ProcessBusinessMappingVo::getBusinessNo).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(businessNoList)){
            throw new RuntimeException("未查询到测量数据");
        }
        MarketingStrategyItemDto itemDto = new MarketingStrategyItemDto();
        itemDto.setStrategyCodeList(businessNoList);
        return marketingStrategyItemService.findByConditions(pageable,itemDto);
    }

    @Override
    public List<MarketingStrategyBudgetSumVo> findApproveProcessSumInfo(String processNo) {
        ProcessBusinessMappingDto processBusinessMappingDto = new ProcessBusinessMappingDto();
        processBusinessMappingDto.setProcessNo(processNo);
        List<ProcessBusinessMappingVo> processBusinessMappingVoList = this.processBusinessMappingService.findMultiByByConditions(processBusinessMappingDto);
        if (CollectionUtils.isEmpty(processBusinessMappingVoList)){
            throw new RuntimeException("流程数据有误！");
        }
        List<String> businessNoList = processBusinessMappingVoList.stream().map(ProcessBusinessMappingVo::getBusinessNo).collect(Collectors.toList());

        MarketingStrategyItemDto itemDto = new MarketingStrategyItemDto();
        itemDto.setStrategyCodeList(businessNoList);
        List<MarketingStrategyBudgetDto> budgetList = marketingStrategyBudgetRepository.listByStrategyItem(itemDto);
        if (CollectionUtils.isEmpty(budgetList)){
            return Lists.newArrayList();
        }
//        预算项目  所属部门名称	年月	累计滚动金额（元）	已使用策略金额（元）	本次策略金额（元）
//        展示发起审批的营销策略所关联预算的预算项目、所属部门名称、年月、累计滚动金额、已使用策略金额、本次策略金额。并按照预算项目+部门名称+年月进行汇总累计滚动金额、已使用策略金额、本次策略金额展示
        List<String> monthBudgetCodeList = budgetList.stream().map(MarketingStrategyBudgetDto::getMonthBudgetCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(monthBudgetCodeList)){
            return Lists.newArrayList();
        }
        List<MonthBudgetVo> monthBudgetList = monthBudgetService.findByCodes(monthBudgetCodeList, null);
        Map<String, MonthBudgetVo> monthBudgetMap = monthBudgetList.stream().collect(Collectors.toMap(MonthBudgetVo::getMonthBudgetCode, Function.identity()));

        Map<String,MarketingStrategyBudgetSumVo> resultMap = Maps.newHashMap();
        for (MarketingStrategyBudgetDto budgetDto : budgetList) {
            MonthBudgetVo monthBudgetVo = monthBudgetMap.get(budgetDto.getMonthBudgetCode());
            String key = monthBudgetVo.getBudgetItemCode() + monthBudgetVo.getOrgCode() + monthBudgetVo.getYearMonthLy();
            MarketingStrategyBudgetSumVo sumVo = resultMap.computeIfAbsent(key,tempKey -> {
                MarketingStrategyBudgetSumVo tempVo = new MarketingStrategyBudgetSumVo();
                tempVo.setBudgetItemCode(monthBudgetVo.getBudgetItemCode());
                tempVo.setBudgetItemName(monthBudgetVo.getBudgetItemName());
                tempVo.setYearMonthLy(monthBudgetVo.getYearMonthLy());
                tempVo.setOrgCode(monthBudgetVo.getOrgCode());
                tempVo.setOrgName(monthBudgetVo.getOrgName());
                tempVo.setAfterFreezeAmount(BigDecimal.ZERO);
                tempVo.setUsedStrategyAmount(BigDecimal.ZERO);
                tempVo.setTotalStrategyAmount(BigDecimal.ZERO);
                return tempVo;
            });
            sumVo.setAfterFreezeAmount(Optional.ofNullable(monthBudgetVo.getAfterFreezeAmount()).orElse(BigDecimal.ZERO).add(sumVo.getAfterFreezeAmount()));
            sumVo.setUsedStrategyAmount(Optional.ofNullable(monthBudgetVo.getUsedStrategyAmount()).orElse(BigDecimal.ZERO).add(sumVo.getUsedStrategyAmount()));
            sumVo.setTotalStrategyAmount(Optional.ofNullable(budgetDto.getUseAmount()).orElse(BigDecimal.ZERO).add(sumVo.getTotalStrategyAmount()));
        }
        return Lists.newArrayList(resultMap.values());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MarketingStrategyOutDto saveSingleMarketingStrategyNoCache(MarketingStrategyOutDto dto) {
        boolean update = !StringUtils.isBlank(dto.getId());
        List<MarketingStrategyItemDto> itemCacheList = dto.getItemCacheList();
        if (BooleanEnum.TRUE.getCapital().equals(dto.getMultipleStrategyAdd())) {
            //多策略新增
            saveMutiMarketingStrategy(dto, itemCacheList);
        } else {
            //单策略新增
            dto.setStrategySignStr(MarketingStrategyConstant.DEFAULT_STRATEGY_SIGN_STR);
            for (MarketingStrategyItemDto marketingStrategyItemDto : itemCacheList) {
                marketingStrategyItemDto.setStrategySignStr(MarketingStrategyConstant.DEFAULT_STRATEGY_SIGN_STR);
                marketingStrategyItemDto.setStrategyName(dto.getStrategyName());
                marketingStrategyItemDto.setStrategyTitle(dto.getStrategyTitle());
                marketingStrategyItemDto.setStrategyDesc(dto.getStrategyDesc());
                marketingStrategyItemDto.setIsCrossMonth(dto.getIsCrossMonth());
            }
            saveMarketingStrategy(dto, itemCacheList);
        }

        //新增修改业务日志
        MarketingStrategyLogEventDto logEventDto = new MarketingStrategyLogEventDto();
        logEventDto.setNewest(dto);
        if (!update) {
            logEventDto.setOriginal(null);
            SerializableBiConsumer<MarketingStrategyLogEventListener, MarketingStrategyLogEventDto> onCreate =
                    MarketingStrategyLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, MarketingStrategyLogEventListener.class, onCreate);
        } else {
            MarketingStrategy oldEntity = marketingStrategyRepository.getById(dto.getId());
            MarketingStrategyDto oldDto = nebulaToolkitService.copyObjectByWhiteList(oldEntity, MarketingStrategyDto.class, HashSet.class, ArrayList.class);
            logEventDto.setOriginal(oldDto);
            SerializableBiConsumer<MarketingStrategyLogEventListener, MarketingStrategyLogEventDto> onUpdate =
                    MarketingStrategyLogEventListener::onUpdate;
            this.nebulaNetEventClient.publish(logEventDto, MarketingStrategyLogEventListener.class, onUpdate);
        }
        return dto;
    }
}

