package com.biz.crm.tpm.business.platform.customer.local.process;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.tpm.business.platform.customer.local.constant.PlatformCustomerConstant;
import com.biz.crm.tpm.business.platform.customer.sdk.dto.PlatformCustomerDto;
import com.biz.crm.tpm.business.platform.customer.sdk.service.PlatformCustomerService;
import com.biz.crm.tpm.business.platform.customer.sdk.vo.PlatformCustomerImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 导入
 *
 * @author: yaoyongming
 * @date: 2022/12/8 19:50
 */
@Slf4j
@Component
public class PlatformCustomerImportsProcess implements ImportProcess<PlatformCustomerImportsVo> {

    @Autowired(required = false)
    private PlatformCustomerService platformCustomerService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, PlatformCustomerImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        log.info("开始导入......");
        Validate.notEmpty(data, "导入数据不能为空！");

        log.info("开始导入校验......");
        List<PlatformCustomerDto> importList = this.validate(data);
        log.info("导入校验结束......");

        log.info("开始执行保存......");
        platformCustomerService.importSave(importList);
        log.info("结束执行保存......");
        return null;
    }

    private List<PlatformCustomerDto> validate(LinkedHashMap<Integer, PlatformCustomerImportsVo> data) {
        List<PlatformCustomerImportsVo> importList = new ArrayList<>(data.values());
        List<PlatformCustomerDto> dtoList = new ArrayList<>(nebulaToolkitService.copyCollectionByWhiteList(importList, PlatformCustomerImportsVo.class, PlatformCustomerDto.class, LinkedHashSet.class, ArrayList.class));

        log.info("一共【"+ dtoList.size() +"】条数据，客户名称【"+ dtoList.get(0).getCustomerName() +"】......");
        log.info("开始非空校验......");
        //非空校验
        dtoList.forEach(e -> {
            Validate.notNull(e, "新增时，对象信息不能为空！");
            e.setId(null);
            Validate.notBlank(e.getPlatformName(), "平台名称，不能为空！");
            Validate.notBlank(e.getShop(), "店铺名称，不能为空！");
            Validate.notBlank(e.getCustomerCode(), "客户编码，不能为空！");
            Validate.notBlank(e.getAppKey(), "appKey，不能为空！");
            Validate.notBlank(e.getAppSecret(), "appSecret，不能为空！");
            Validate.notBlank(e.getUrl(), "授权url，不能为空！");
        });
        log.info("结束非空校验......");

        log.info("开始查询数据字典......");
        // 查询数据字典
        List<DictDataVo> dictDataVos = dictDataVoService.findByDictTypeCode(PlatformCustomerConstant.DICT_PLATFORM);
        log.info("结束查询数据字典......");
        Map<String, CustomerVo> customerVoMap = new HashMap<>();
        log.info("客户set开始......");
        Set<String> customerCodeSet = dtoList.stream().map(PlatformCustomerDto::getCustomerCode).filter(Objects::nonNull).collect(Collectors.toSet());
        log.info("客户set结束......");
        log.info("客户获取开始......");
        List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerCodeSet));
        log.info("客户获取结束......");
        log.info("客户封装开始......");
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
        } else {
            throw new IllegalArgumentException("未找到任意客户，请检查！");
        }
        log.info("客户封装结束......");

        log.info("开始关联数据校验......");
        //关联数据校验
        for (PlatformCustomerDto dto : dtoList) {
            if (customerVoMap.containsKey(dto.getCustomerCode())) {
                dto.setCustomerName(customerVoMap.get(dto.getCustomerCode()).getCustomerName());
            } else {
                throw new IllegalArgumentException("客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
            }
            DictDataVo dictDataVo = dictDataVos.stream().filter(e -> e.getDictValue().equals(dto.getPlatformName())).findFirst().orElse(null);
            if (dictDataVo != null) {
                dto.setPlatformCode(dictDataVo.getDictCode());
            } else {
                throw new IllegalArgumentException("平台名称【" + dto.getPlatformName() + "】错误，请检查！");
            }
        }
        log.info("结束关联数据校验......");

        log.info("开始重复校验......");
        //重复校验
        Set<String> setRepeat = dtoList.stream().map(e -> e.getPlatformCode()
                + "_" + e.getCustomerCode()
                + "_" + e.getShop()
                + "_" + e.getAppKey()
                + "_" + e.getAppSecret()
                + "_" + e.getUrl()).collect(Collectors.toSet());
        Validate.isTrue(setRepeat.size() == dtoList.size() , "存在重复的数据不允许保存");
        log.info("结束重复校验......");
        log.info("开始数据库重复校验......");
        dtoList.forEach(e -> platformCustomerService.validateRepeatabilityOnCreate(e));
        log.info("结束数据库重复校验......");
        return dtoList;
    }

    @Override
    public Class<PlatformCustomerImportsVo> findCrmExcelVoClass() {
        return PlatformCustomerImportsVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_PLATFORM_CUSTOMER_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM平台客户导入";
    }

    @Override
    public Integer getBatchCount() {
        return 10000;
    }
}
