package com.biz.crm.tpm.business.platform.customer.local.service.internal;


import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.tpm.business.platform.customer.local.constant.PlatformCustomerConstant;
import com.biz.crm.tpm.business.platform.customer.local.repository.PlatformCustomerRepository;
import com.biz.crm.tpm.business.platform.customer.local.entity.PlatformCustomer;
import com.biz.crm.tpm.business.platform.customer.sdk.dto.PlatformCustomerDto;
import com.biz.crm.tpm.business.platform.customer.sdk.dto.log.PlatformCustomerLogEventDto;
import com.biz.crm.tpm.business.platform.customer.sdk.event.log.PlatformCustomerLogEventListener;
import com.biz.crm.tpm.business.platform.customer.sdk.service.PlatformCustomerService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 平台客户(PlatformCustomer)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-12-05 15:44:08
 */
@Service("platformCustomerService")
public class PlatformCustomerServiceImpl implements PlatformCustomerService {

    @Autowired(required = false)
    private PlatformCustomerRepository platformCustomerRepository;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public PlatformCustomerDto findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        PlatformCustomer entity = this.platformCustomerRepository.getById(id);
        if (entity == null) {
            return null;
        }
        return nebulaToolkitService.copyObjectByWhiteList(entity, PlatformCustomerDto.class, HashSet.class, ArrayList.class);
    }

    /**
     * 新增数据
     *
     * @param platformCustomer 实体对象
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public PlatformCustomerDto create(PlatformCustomerDto platformCustomer) {
        this.createValidate(platformCustomer);

        // redis生成料编码code
        List<String> codeList = this.generateCodeService.generateCode(PlatformCustomerConstant.PREFIX_CODE + DateFormatUtils.format(new Date(), "yyyyMMdd") , 1, 5, 2, TimeUnit.DAYS);
        platformCustomer.setPlatformCustomerCode(codeList.get(0));
        platformCustomer.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        platformCustomer.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        platformCustomer.setTenantCode(TenantUtils.getTenantCode());



        this.platformCustomerRepository.saveOrUpdate(nebulaToolkitService.copyObjectByWhiteList(platformCustomer, PlatformCustomer.class, HashSet.class, ArrayList.class));

        //新增业务日志
        PlatformCustomerLogEventDto logEventDto = new PlatformCustomerLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(platformCustomer);
        SerializableBiConsumer<PlatformCustomerLogEventListener, PlatformCustomerLogEventDto> onCreate =
                PlatformCustomerLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, PlatformCustomerLogEventListener.class, onCreate);
        return platformCustomer;
    }

    /**
     * 修改数据
     *
     * @param platformCustomer 实体对象
     * @return 修改结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public PlatformCustomerDto update(PlatformCustomerDto platformCustomer) {
        this.updateValidate(platformCustomer);

        platformCustomer.setPlatformCustomerCode(null);
        platformCustomer.setTenantCode(null);

        PlatformCustomer entityOld = platformCustomerRepository.getById(platformCustomer.getId());

        Validate.notNull(entityOld, "未找到修改的数据");

        this.platformCustomerRepository.saveOrUpdate(nebulaToolkitService.copyObjectByWhiteList(platformCustomer, PlatformCustomer.class, HashSet.class, ArrayList.class));

        //编辑业务日志
        PlatformCustomerLogEventDto logEventDto = new PlatformCustomerLogEventDto();
        logEventDto.setOriginal(this.nebulaToolkitService.copyObjectByWhiteList(entityOld, PlatformCustomerDto.class, LinkedHashSet.class, ArrayList.class));
        logEventDto.setNewest(platformCustomer);
        SerializableBiConsumer<PlatformCustomerLogEventListener, PlatformCustomerLogEventDto> onUpdate =
                PlatformCustomerLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, PlatformCustomerLogEventListener.class, onUpdate);
        return platformCustomer;
    }

    /**
     * 批量启用
     *
     * @param ids
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void enableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待启用的数据主键不能为空");

        List<PlatformCustomer> list = this.platformCustomerRepository.listByIds(ids);
        Validate.notEmpty(list, "待启用的数据不存在或已删除!");

        platformCustomerRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);

        //启用业务日志
        Collection<PlatformCustomerDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(list,
                PlatformCustomer.class, PlatformCustomerDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<PlatformCustomerLogEventListener, PlatformCustomerLogEventDto> onEnable =
                PlatformCustomerLogEventListener::onEnable;
        for (PlatformCustomerDto dto : dtoList) {
            PlatformCustomerLogEventDto logEventDto = new PlatformCustomerLogEventDto();
            logEventDto.setNewest(dto);
            dto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            logEventDto.setOriginal(dto);
            this.nebulaNetEventClient.publish(logEventDto, PlatformCustomerLogEventListener.class, onEnable);
        }
    }

    /**
     * 批量禁用
     *
     * @param ids
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void disableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待禁用的数据主键不能为空");

        List<PlatformCustomer> configList = this.platformCustomerRepository.listByIds(ids);
        Validate.notEmpty(configList, "待禁用的数据不存在或已删除!");

        platformCustomerRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);

        //禁用业务日志
        Collection<PlatformCustomerDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(configList,
                PlatformCustomer.class, PlatformCustomerDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<PlatformCustomerLogEventListener, PlatformCustomerLogEventDto> onDisable =
                PlatformCustomerLogEventListener::onDisable;
        for (PlatformCustomerDto dto : dtoList) {
            PlatformCustomerLogEventDto logEventDto = new PlatformCustomerLogEventDto();
            logEventDto.setNewest(dto);
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setOriginal(dto);
            this.nebulaNetEventClient.publish(logEventDto, PlatformCustomerLogEventListener.class, onDisable);
        }
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<PlatformCustomer> list = this.platformCustomerRepository.listByIds(idList);
        Validate.notEmpty(list, "根据提供的主键集合信息，未能获取到相应数据");

        this.platformCustomerRepository.removePlatformCustomerByIds(idList);

        //删除业务日志
        Collection<PlatformCustomerDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(list,
                PlatformCustomer.class, PlatformCustomerDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<PlatformCustomerLogEventListener, PlatformCustomerLogEventDto> onDelete =
                PlatformCustomerLogEventListener::onDelete;
        for (PlatformCustomerDto dto : dtoList) {
            PlatformCustomerLogEventDto logEventDto = new PlatformCustomerLogEventDto();
            logEventDto.setOriginal(dto);
            this.nebulaNetEventClient.publish(logEventDto, PlatformCustomerLogEventListener.class, onDelete);
        }
    }

    /**
     * 批量导入
     *
     * @param importList
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void importSave(List<PlatformCustomerDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        // redis生成料编码code
        List<String> codeList = this.generateCodeService.generateCode(PlatformCustomerConstant.PREFIX_CODE + DateFormatUtils.format(new Date(), "yyyyMMdd") , importList.size(), 5, 2, TimeUnit.DAYS);
        for (int i = 0; i < importList.size(); i++) {
            PlatformCustomerDto dto = importList.get(i);
            dto.setPlatformCustomerCode(codeList.get(i));
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setTenantCode(TenantUtils.getTenantCode());
        }

        this.platformCustomerRepository.saveBatch(nebulaToolkitService.copyCollectionByWhiteList(importList, PlatformCustomerDto.class, PlatformCustomer.class, HashSet.class, ArrayList.class));

        //新增业务日志
        importList.forEach(dto -> {
            PlatformCustomerLogEventDto logEventDto = new PlatformCustomerLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<PlatformCustomerLogEventListener, PlatformCustomerLogEventDto> onCreate =
                    PlatformCustomerLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, PlatformCustomerLogEventListener.class, onCreate);
        });
    }

    /**
     * 创建验证
     *
     * @param platformCustomer
     */
    @Override
    public void createValidate(PlatformCustomerDto platformCustomer) {
        Validate.notNull(platformCustomer, "新增时，对象信息不能为空！");
        platformCustomer.setId(null);
        commonValidate(platformCustomer);
        validateRepeatabilityOnCreate(platformCustomer);
    }

    /**
     * 修改验证
     *
     * @param platformCustomer
     */
    private void updateValidate(PlatformCustomerDto platformCustomer) {
        Validate.notNull(platformCustomer, "修改时，对象信息不能为空！");
        Validate.notBlank(platformCustomer.getId(), "id，不能为空！");
        commonValidate(platformCustomer);
        validateRepeatabilityOnUpdate(platformCustomer);
    }

    /**
     * 创建校验重复数据
     *
     * @param dto
     * @return
     */
    @Override
    public void validateRepeatabilityOnCreate(PlatformCustomerDto dto) {
        List<PlatformCustomer> list = platformCustomerRepository.findOnCreate(dto);
        Validate.isTrue(CollectionUtils.isEmpty(list), "存在重复的数据不允许保存");
    }

    /**
     * 修改校验重复数据
     *
     * @param dto
     * @return
     */
    private void validateRepeatabilityOnUpdate(PlatformCustomerDto dto) {
        List<PlatformCustomer> list = platformCustomerRepository.findOnUpdate(dto);
        Validate.isTrue(CollectionUtils.isEmpty(list), "存在重复的数据不允许保存");
    }

    /**
     * 公共校验
     *
     * @param platformCustomer
     */
    private void commonValidate(PlatformCustomerDto platformCustomer) {
        Validate.notBlank(platformCustomer.getPlatformCode(), "平台编码，不能为空！");
        Validate.notBlank(platformCustomer.getPlatformName(), "平台名称，不能为空！");
        Validate.notBlank(platformCustomer.getShop(), "店铺名称，不能为空！");
        Validate.notBlank(platformCustomer.getCustomerCode(), "客户编码，不能为空！");
        Validate.notBlank(platformCustomer.getCustomerName(), "客户名称，不能为空！");
        Validate.notBlank(platformCustomer.getAppKey(), "appKey，不能为空！");
        Validate.notBlank(platformCustomer.getAppSecret(), "appSecret，不能为空！");
        Validate.notBlank(platformCustomer.getUrl(), "授权url，不能为空！");
    }
}

