package com.biz.crm.tpm.business.settlement.manage.local.service.imports;

import com.aliyun.openservices.shade.com.google.common.collect.Sets;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateStringDealUtil;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.platform.expense.form.sdk.dto.PlatformExpenseFormDto;
import com.biz.crm.tpm.business.platform.expense.form.sdk.service.PlatformExpenseFormService;
import com.biz.crm.tpm.business.platform.product.relation.sdk.dto.PlatformProductRelationDto;
import com.biz.crm.tpm.business.platform.product.relation.sdk.service.PlatformProductRelationService;
import com.biz.crm.tpm.business.platform.product.relation.sdk.vo.PlatformProductRelationVo;
import com.biz.crm.tpm.business.settlement.manage.local.entity.TpmSettlementManage;
import com.biz.crm.tpm.business.settlement.manage.local.repository.TpmSettlementManageRepository;
import com.biz.crm.tpm.business.settlement.manage.sdk.constant.SettlementManageConstant;
import com.biz.crm.tpm.business.settlement.manage.sdk.enums.TpmSettlementSourceEnum;
import com.biz.crm.tpm.business.settlement.manage.sdk.vo.imports.JDmallSettlementImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * TPM结算单管理-京东商城导入
 *
 * @author duyiran
 * @create 2022-11-02 19:47
 */
@Slf4j
@Component
public class JDmallSettlementImportsProcess implements ImportProcess<JDmallSettlementImportsVo> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private PlatformExpenseFormService platformExpenseFormService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    PlatformProductRelationService platformProductRelationService;

    @Autowired(required = false)
    private TpmSettlementManageRepository tpmSettlementManageRepository;

    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, JDmallSettlementImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notEmpty(data, "导入数据不能为空！");
        try {
            this.baseValidate(data);
            this.validateRepeatAndSave(data);
        } catch (Exception e) {
            log.error(e.getMessage());
            throw e;
        }
        return null;
    }

    /**
     * 校验数据
     *
     * @param data data
     * @author duyiran
     * @date 2022-10-29 15:04
     **/
    private void baseValidate(LinkedHashMap<Integer, JDmallSettlementImportsVo> data) {
        Map<String, String> mdmBusinessFormatMap = new HashMap<>();
        Map<String, CustomerVo> customerVoMap = new HashMap<>();
        Map<String, PlatformProductRelationVo> platformProductRelationVoMap = new HashMap<>();
        Map<String, PlatformExpenseFormDto> expenseFormDtoMap = new HashMap<>();
        mdmBusinessFormatMap = dictToolkitService.findConvertMapByDictTypeCode(SettlementManageConstant.MDM_BUSINESS_FORMAT);
        Set<String> customerCodeSet = Sets.newHashSet();
        Set<String> expenseItemSet = Sets.newHashSet();
        Set<String> productCodeSet = Sets.newHashSet();
        for (int i = 0; i < data.size(); i++) {
            JDmallSettlementImportsVo importsVo = data.get(i);
            Validate.notBlank(importsVo.getCode(), "结算单号不能为空！");
            Validate.notBlank(importsVo.getSubCode(), "单据编号不能为空！");
            Validate.notBlank(importsVo.getPlatformProductCode(), "SKU编码不能为空！");
            productCodeSet.add(importsVo.getPlatformProductCode());
            Validate.notBlank(importsVo.getExpenseItem(), "应付帐单据类型不能为空！");
            expenseItemSet.add(importsVo.getExpenseItem());
            String customerCode = "";
            String mdgCode = importsVo.getMdgCode();
            Validate.notBlank(mdgCode, "客户编码不能为空！");
            customerCode += mdgCode;
            String salesOrgCode = importsVo.getSalesOrgCode();
            Validate.notBlank(salesOrgCode, "销售机构编码不能为空！");
            customerCode += salesOrgCode;
            String channelCode = importsVo.getChannelCode();
            Validate.notBlank(channelCode, "渠道编码不能为空！");
            customerCode += channelCode;
            String businessFormatCode = importsVo.getBusinessFormatCode();
            Validate.notBlank(businessFormatCode, "业态不能为空！");
            // 业态
            if (!StringUtils.isEmpty(businessFormatCode)) {
                if (!mdmBusinessFormatMap.containsKey(businessFormatCode)) {
                    throw new RuntimeException("业态" + businessFormatCode + "错误，请检查！");
                } else {
                    importsVo.setBusinessFormatCode(mdmBusinessFormatMap.get(businessFormatCode));
                }
            }
            customerCode += importsVo.getBusinessFormatCode();
            customerCodeSet.add(customerCode);
            importsVo.setCustomerCode(customerCode);
            try {
                importsVo.setQuantity(new Integer(importsVo.getQuantityStr()));
                importsVo.setPrice(new BigDecimal(importsVo.getPriceStr()));
            } catch (Exception e) {
                log.error("导入报错:{}", e.getMessage());
                throw new IllegalArgumentException("数量、单价不能为空或格式错误！");
            }
            try {
                importsVo.setAmount(new BigDecimal(importsVo.getAmountStr()));
            } catch (Exception e) {
                log.error("导入报错:{}", e.getMessage());
                throw new IllegalArgumentException("金额不能为空或格式错误！");
            }
            try {
                DateStringDealUtil.validateDateStrAndSet(importsVo.getBusinessTimeStr(), "业务发生时间", true, DateUtil.DEFAULT_DATE_ALL_PATTERN, importsVo::setBusinessTime);
            } catch (Exception e) {
                throw new IllegalArgumentException(e.getMessage());
            }
            importsVo.setYearMonthStr(DateUtil.dateToStr(DateUtil.date_yyyy_MM, importsVo.getBusinessTime()));
        }
        //客户
        if (!CollectionUtils.isEmpty(customerCodeSet)) {
            List<CustomerVo> customerVos = customerVoService.findByCustomerCodes(new ArrayList<>(customerCodeSet));
            if (!CollectionUtils.isEmpty(customerVos)) {
                customerVoMap = customerVos.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
            }
        }
        // 产品
        if (!CollectionUtils.isEmpty(productCodeSet) && !CollectionUtils.isEmpty(customerCodeSet)) {
            PlatformProductRelationDto platformProductRelationDto = new PlatformProductRelationDto();
            platformProductRelationDto.setPlatformProductCodeSet(productCodeSet);
            platformProductRelationDto.setCustomerCodeSet(customerCodeSet);
            List<PlatformProductRelationVo> platformVoList = platformProductRelationService.findListByConditions(platformProductRelationDto);
            if (!CollectionUtils.isEmpty(platformVoList)) {
                platformProductRelationVoMap = platformVoList.stream().collect(Collectors.toMap(e -> e.getCustomerCode() + e.getPlatformProductCode(), Function.identity(), (e1, e2) -> e1));
            }
        }
        // 费用项
        if (!CollectionUtils.isEmpty(productCodeSet)) {
            List<PlatformExpenseFormDto> expenseFormDtoList = platformExpenseFormService.findListByExpenseNameSet(expenseItemSet);
            if (!CollectionUtils.isEmpty(expenseFormDtoList)) {
                expenseFormDtoMap = expenseFormDtoList.stream().collect(Collectors.toMap(PlatformExpenseFormDto::getExpensesName, Function.identity(), (e1, e2) -> e1));
            }
        }
        for (int i = 0; i < data.size(); i++) {
            JDmallSettlementImportsVo importsVo = data.get(i);
            CustomerVo customerVo = customerVoMap.get(importsVo.getCustomerCode());
            PlatformProductRelationVo productVo = platformProductRelationVoMap.get(importsVo.getCustomerCode() + importsVo.getPlatformProductCode());
            PlatformExpenseFormDto platformExpenseFormDto = expenseFormDtoMap.get(importsVo.getExpenseItem());
            Validate.notNull(customerVo, "根据MDG客户编码+销售机构编码+渠道编码+业态编码未找到对应客户！");
            importsVo.setCustomerName(customerVo.getCustomerName());
            importsVo.setSalesOrgCode(customerVo.getSalesInstitutionErpCode());
            importsVo.setSalesOrgName(customerVo.getSalesInstitutionName());
            importsVo.setChannelCode(customerVo.getCustomerChannelCode());
            importsVo.setChannelName(customerVo.getCustomerChannelName());
            Validate.notNull(productVo, "根据商品编码未找到对应产品！");
            importsVo.setProductName(productVo.getProductName());
            importsVo.setProductCode(productVo.getProductCode());
            importsVo.setPlatformProductCode(productVo.getPlatformProductCode());
            importsVo.setPlatformProductName(productVo.getPlatformProductName());
            Validate.notNull(platformExpenseFormDto, "根据应付帐单据类型未找到对应活动形式！");
            importsVo.setActivityFormCode(platformExpenseFormDto.getActivityFormCode());
            importsVo.setActivityFormName(platformExpenseFormDto.getActivityFormName());
            importsVo.setSourceCode(TpmSettlementSourceEnum.JD_MALL.getCode());
        }
    }

    /**
     * 验重，成功后保存
     *
     * @param data data
     */
    private void validateRepeatAndSave(LinkedHashMap<Integer, JDmallSettlementImportsVo> data) {
        Map<String, Integer> repeatKeyToIndexMap = Maps.newHashMap();
        data.forEach((index, importsVo) -> {
            //行内验重 repeatKey
            String key = TpmSettlementSourceEnum.JD_MALL.getCode() + importsVo.getCode() + importsVo.getSubCode() + importsVo.getCustomerCode() + importsVo.getPlatformProductCode() + importsVo.getExpenseItem();
            if (repeatKeyToIndexMap.containsKey(key)) {
                Integer lastIndex = repeatKeyToIndexMap.get(key);
                throw new IllegalArgumentException("该行结算单号+单据编号+客户编码+后端商品编码+应付帐单据类型与第【" + lastIndex + "】行重复");
            } else {
                repeatKeyToIndexMap.put(key, index);
                importsVo.setVerifyId(key);
            }
        });
        List<TpmSettlementManage> importList = (List<TpmSettlementManage>) nebulaToolkitService.copyCollectionByWhiteList(data.values(), JDmallSettlementImportsVo.class, TpmSettlementManage.class, LinkedHashSet.class, ArrayList.class);
        List<String> repeatKeyList = new ArrayList<>(repeatKeyToIndexMap.keySet());
        boolean isLock = redisLockService.batchLock(SettlementManageConstant.SETTLEMENT_MANAGE_LOCK, repeatKeyList, TimeUnit.MINUTES, SettlementManageConstant.DEFAULT_LOCK_TIME);
        Validate.isTrue(isLock, "加锁失败");
        //数据库验重
        Set<String> repeatSet = tpmSettlementManageRepository.repeatValidateBatch(repeatKeyList);
        if (CollectionUtils.isNotEmpty(repeatSet)) {
            data.forEach((index, importsVo) -> {
                //数据库验重
                if (repeatSet.contains(importsVo.getVerifyId())) {
                    throw new IllegalArgumentException("结算单号+单据编号+客户编码+后端商品编码+应付帐单据类型已有现存数据，请重新填写！");
                }
            });
        }
        tpmSettlementManageRepository.bulkPreservation(importList);
        try {
            redisLockService.batchUnLock(SettlementManageConstant.SETTLEMENT_MANAGE_LOCK, repeatKeyList);
        } catch (Exception ignore) {}
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<JDmallSettlementImportsVo> findCrmExcelVoClass() {
        return JDmallSettlementImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_SETTLEMENT_JD_MALL_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "TPM结算单管理-京东商城导入";
    }
}
