package com.biz.crm.tpm.business.settlement.manage.local.service.internal;


import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.tpm.business.settlement.manage.local.repository.TpmSettlementManageRepository;
import com.biz.crm.tpm.business.settlement.manage.local.entity.TpmSettlementManage;
import com.biz.crm.tpm.business.settlement.manage.sdk.constant.SettlementManageConstant;
import com.biz.crm.tpm.business.settlement.manage.sdk.dto.TpmSettlementManageDto;
import com.biz.crm.tpm.business.settlement.manage.sdk.dto.log.TpmSettlementManageLogEventDto;
import com.biz.crm.tpm.business.settlement.manage.sdk.event.log.TpmSettlementManageLogEventListener;
import com.biz.crm.tpm.business.settlement.manage.sdk.service.TpmSettlementManageService;
import com.biz.crm.tpm.business.settlement.manage.sdk.vo.TpmSettlementManageVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * TPM-结算单管理(TpmSettlementManage)表服务实现类
 *
 * @author duyiran
 * @since 2022-11-01 16:38:48
 */
@Slf4j
@Service("tpmSettlementManageService")
public class TpmSettlementManageServiceImpl implements TpmSettlementManageService {

    @Autowired(required = false)
    private TpmSettlementManageRepository tpmSettlementManageRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      实体对象
     * @return
     */
    @Override
    public Page<TpmSettlementManageVo> findByConditions(Pageable pageable, TpmSettlementManageDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmSettlementManageDto();
        }
        return this.tpmSettlementManageRepository.findByConditions(pageable, dto);
    }

    /**
     * 客户+产品+年月,统计销售数量
     *
     * @param dto dto对象
     * @return
     */
    @Override
    public BigDecimal getAmount(TpmSettlementManageDto dto) {
        if (Objects.isNull(dto) || Strings.isEmpty(dto.getCustomerCode()) || Strings.isEmpty(dto.getExpenseItem()) ||
                Objects.isNull(dto.getBusinessStartTime()) || Objects.isNull(dto.getBusinessEndTime())) {
            return BigDecimal.ZERO;
        }
        return this.tpmSettlementManageRepository.getAmount(dto);
    }

    /**
     * 客户+产品+年月,统计销售数量
     *
     * @param dto dto对象
     * @return
     */
    @Override
    public Integer getQuantity(TpmSettlementManageDto dto) {
        if (Objects.isNull(dto) || Strings.isEmpty(dto.getCustomerCode()) || Strings.isEmpty(dto.getExpenseItem()) ||
                Objects.isNull(dto.getBusinessStartTime()) || Objects.isNull(dto.getBusinessEndTime())) {
            return 0;
        }
        return this.tpmSettlementManageRepository.getQuantity(dto);
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        this.tpmSettlementManageRepository.lambdaUpdate().in(TpmSettlementManage::getId, idList)
                .set(TpmSettlementManage::getDelFlag, DelFlagStatusEnum.DELETE.getCode()).update();
    }

    /**
     * 导入保存
     *
     * @param importList
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void importSave(List<TpmSettlementManageDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        Collection<TpmSettlementManage> entities = this.nebulaToolkitService.copyCollectionByWhiteList(importList, TpmSettlementManageDto.class, TpmSettlementManage.class, LinkedHashSet.class, ArrayList.class);
        this.tpmSettlementManageRepository.bulkPreservation(new ArrayList<>(entities));

        //日志
        if (importList.size() > 10000) {
            while (importList.size() > 0) {
                int removeSize = Math.min(importList.size(), 50);
                List<TpmSettlementManageDto> removeList = importList.subList(0, removeSize);
                // 发送MQ消息
                MqMessageVo mqMessage = new MqMessageVo();
                //数据
                mqMessage.setMsgBody(JSON.toJSONString(removeList));
                //MQ标签
                mqMessage.setTag(SettlementManageConstant.SETTLEMENT_MANAGE_CREATE_LOG_TAG);
                rocketMqProducer.sendMqMsg(mqMessage);
                //截断
                importList.removeAll(removeList);
            }
        } else {
            this.logCreate(importList);
        }
    }

    /**
     * 保存新增日志
     *
     * @param dtoList
     */
    @Override
    public void logCreate(List<TpmSettlementManageDto> dtoList) {
        dtoList.forEach(dto -> {
            TpmSettlementManageLogEventDto logEventDto = new TpmSettlementManageLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<TpmSettlementManageLogEventListener, TpmSettlementManageLogEventDto> onCreate =
                    TpmSettlementManageLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, TpmSettlementManageLogEventListener.class, onCreate);
        });
    }

    /**
     * 平台结算管理相关关键指标查询费用，汇总，客户+产品+时间段/客户+时间段
     *
     * @param dto
     * @return
     */
    @Override
    public BigDecimal getAmountByActivityFormCode(TpmSettlementManageDto dto) {
        return tpmSettlementManageRepository.getAmountByActivityFormCode(dto);
    }

}

