package com.biz.crm.wechatpay.v3.service.impl;

import com.biz.crm.wechatpay.util.HttpServletUtil;
import com.biz.crm.wechatpay.v3.model.*;
import com.biz.crm.wechatpay.v3.service.WechatPayV3TransferBatchService;
import com.wechat.pay.java.core.Config;
import com.wechat.pay.java.core.cipher.PrivacyDecryptor;
import com.wechat.pay.java.core.cipher.PrivacyEncryptor;
import com.wechat.pay.java.core.exception.HttpException;
import com.wechat.pay.java.core.exception.MalformedMessageException;
import com.wechat.pay.java.core.exception.ServiceException;
import com.wechat.pay.java.core.exception.ValidationException;
import com.wechat.pay.java.core.http.*;
import com.wechat.pay.java.core.notification.NotificationConfig;
import com.wechat.pay.java.core.notification.NotificationParser;
import com.wechat.pay.java.core.notification.RequestParam;

import javax.servlet.http.HttpServletRequest;

import static com.wechat.pay.java.core.http.UrlEncoder.urlEncode;
import static com.wechat.pay.java.core.util.GsonUtil.toJson;
import static java.util.Objects.requireNonNull;

/**
 * V3转账老版本
 * 接口文档地址：https://pay.weixin.qq.com/doc/v3/merchant/4012064660
 */
public class WechatPayV3TransferBatchServiceImpl implements WechatPayV3TransferBatchService {

    //通过微信批次单号查询批次单URL
    private static final String GET_TRANSFER_BATCH_BY_BATCH_ID_URL = "https://api.mch.weixin.qq.com/v3/transfer/batches/batch-id/%s";
    //通过商家批次单号查询批次单URL
    private static final String GET_TRANSFER_BATCH_BY_OUT_BATCH_NO_URL = "https://api.mch.weixin.qq.com/v3/transfer/batches/out-batch-no/%s";
    //发起转账URL
    private static final String TRANSFER_BATCH_URL = "https://api.mch.weixin.qq.com/v3/transfer/batches";
    //通过微信明细单号查询明细单URL
    private static final String GET_DETAIL_BY_BATCH_ID_URL = "https://api.mch.weixin.qq.com/v3/transfer/batches/batch-id/%s/details/detail-id/%s";
    //通过商家明细单号查询明细单URL
    private static final String GET_DETAIL_BY_BATCH_NO_URL = "https://api.mch.weixin.qq.com/v3/transfer/batches/out-batch-no/%s/details/out-detail-no/%s";

    private final HttpClient httpClient;
    private final PrivacyEncryptor encryptor;
    private final PrivacyDecryptor decryptor;
    private final NotificationParser parser;

    private WechatPayV3TransferBatchServiceImpl(
            HttpClient httpClient,
            PrivacyEncryptor encryptor,
            PrivacyDecryptor decryptor,
            NotificationParser parser) {
        this.httpClient = requireNonNull(httpClient);
        this.encryptor = requireNonNull(encryptor);
        this.decryptor = requireNonNull(decryptor);
        this.parser=requireNonNull(parser);
    }

    /**
     * WechatPayV3TransferBatchService构造器
     */
    public static class Builder {

        private HttpClient httpClient;
        private PrivacyEncryptor encryptor;
        private PrivacyDecryptor decryptor;
        private NotificationParser parser;

        public Builder config(Config config) {
            this.httpClient = new DefaultHttpClientBuilder().config(config).build();
            this.encryptor = config.createEncryptor();
            this.decryptor = config.createDecryptor();
            this.parser= new NotificationParser((NotificationConfig) config);
            return this;
        }

        public Builder httpClient(HttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }

        public Builder encryptor(PrivacyEncryptor encryptor) {
            this.encryptor = encryptor;
            return this;
        }

        public Builder decryptor(PrivacyDecryptor decryptor) {
            this.decryptor = decryptor;
            return this;
        }
        public Builder parser(NotificationParser parser) {
            this.parser = parser;
            return this;
        }

        public WechatPayV3TransferBatchService build() {
            return new WechatPayV3TransferBatchServiceImpl(httpClient, encryptor, decryptor,parser);
        }
    }

    /**
     * 通过微信批次单号查询批次单
     *
     * @param realRequest 请求参数
     * @return TransferBatchEntity
     * @throws HttpException             发送HTTP请求失败。例如构建请求参数失败、发送请求失败、I/O错误等。包含请求信息。
     * @throws ValidationException       发送HTTP请求成功，验证微信支付返回签名失败。
     * @throws ServiceException          发送HTTP请求成功，服务返回异常。例如返回状态码小于200或大于等于300。
     * @throws MalformedMessageException 服务返回成功，content-type不为application/json、解析返回体失败。
     */
    public TransferBatchResponse getTransferBatchByNo(GetTransferBatchByNoRequest realRequest) {
        //1.格式化url
        String requestPath = String.format(GET_TRANSFER_BATCH_BY_BATCH_ID_URL, urlEncode(realRequest.getBatchId()));
        //2.添加 query param
        QueryParameter queryParameter = new QueryParameter();
        if (realRequest.getNeedQueryDetail() != null) {
            queryParameter.add("need_query_detail", urlEncode(realRequest.getNeedQueryDetail().toString()));
        }
        if (realRequest.getOffset() != null) {
            queryParameter.add("offset", urlEncode(realRequest.getOffset().toString()));
        }
        if (realRequest.getLimit() != null) {
            queryParameter.add("limit", urlEncode(realRequest.getLimit().toString()));
        }
        if (realRequest.getDetailStatus() != null && realRequest.getNeedQueryDetail()) {
            queryParameter.add("detail_status", urlEncode(realRequest.getDetailStatus()));
        }
        requestPath = requestPath.concat(queryParameter.getQueryStr());

        //3.构建请求
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.GET)
                        .url(requestPath)
                        .headers(headers)
                        .build();
        //4.执行请求
        HttpResponse<TransferBatchResponse> httpResponse =
                httpClient.execute(httpRequest, TransferBatchResponse.class);
        return httpResponse.getServiceResponse();
    }

    /**
     * 通过商家批次单号查询批次单
     *
     * @param realRequest 请求参数
     * @return TransferBatchEntity
     * @throws HttpException             发送HTTP请求失败。例如构建请求参数失败、发送请求失败、I/O错误等。包含请求信息。
     * @throws ValidationException       发送HTTP请求成功，验证微信支付返回签名失败。
     * @throws ServiceException          发送HTTP请求成功，服务返回异常。例如返回状态码小于200或大于等于300。
     * @throws MalformedMessageException 服务返回成功，content-type不为application/json、解析返回体失败。
     */
    public TransferBatchResponse getTransferBatchByOutNo(GetTransferBatchByOutNoRequest realRequest) {
        //1.格式化url
        String requestPath = String.format(GET_TRANSFER_BATCH_BY_OUT_BATCH_NO_URL, urlEncode(realRequest.getOutBatchNo()));
        //2.添加 query param
        QueryParameter queryParameter = new QueryParameter();
        if (realRequest.getNeedQueryDetail() != null) {
            queryParameter.add("need_query_detail", urlEncode(realRequest.getNeedQueryDetail().toString()));
        }
        if (realRequest.getOffset() != null) {
            queryParameter.add("offset", urlEncode(realRequest.getOffset().toString()));
        }
        if (realRequest.getLimit() != null) {
            queryParameter.add("limit", urlEncode(realRequest.getLimit().toString()));
        }
        if (realRequest.getDetailStatus() != null && realRequest.getNeedQueryDetail()) {
            queryParameter.add("detail_status", urlEncode(realRequest.getDetailStatus()));
        }
        requestPath = requestPath.concat(queryParameter.getQueryStr());

        //3.构建请求
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.GET)
                        .url(requestPath)
                        .headers(headers)
                        .build();
        //4.执行请求
        HttpResponse<TransferBatchResponse> httpResponse =
                httpClient.execute(httpRequest, TransferBatchResponse.class);
        return httpResponse.getServiceResponse();
    }

    /**
     * V3老版本发起商家转账
     *
     * @param request 请求参数
     * @return InitiateBatchTransferResponse
     * @throws HttpException             发送HTTP请求失败。例如构建请求参数失败、发送请求失败、I/O错误等。包含请求信息。
     * @throws ValidationException       发送HTTP请求成功，验证微信支付返回签名失败。
     * @throws ServiceException          发送HTTP请求成功，服务返回异常。例如返回状态码小于200或大于等于300。
     * @throws MalformedMessageException 服务返回成功，content-type不为application/json、解析返回体失败。
     */
    public InitiateBatchTransferResponse batchTransfer(InitiateBatchTransferRequest request) {
        // 加密敏感信息
        InitiateBatchTransferRequest realRequest = request.cloneWithCipher(encryptor::encrypt);
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.WECHAT_PAY_SERIAL, encryptor.getWechatpaySerial());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.POST)
                        .url(TRANSFER_BATCH_URL)
                        .headers(headers)
                        .body(createRequestBody(realRequest))
                        .build();
        HttpResponse<InitiateBatchTransferResponse> httpResponse =
                httpClient.execute(httpRequest, InitiateBatchTransferResponse.class);
        return httpResponse.getServiceResponse();
    }

    private RequestBody createRequestBody(Object request) {
        return new JsonRequestBody.Builder().body(toJson(request)).build();
    }

    /**
     * 通过微信明细单号查询明细单
     *
     * @param request 请求参数
     * @return TransferDetailVo
     * @throws HttpException             发送HTTP请求失败。例如构建请求参数失败、发送请求失败、I/O错误等。包含请求信息。
     * @throws ValidationException       发送HTTP请求成功，验证微信支付返回签名失败。
     * @throws ServiceException          发送HTTP请求成功，服务返回异常。例如返回状态码小于200或大于等于300。
     * @throws MalformedMessageException 服务返回成功，content-type不为application/json、解析返回体失败。
     */
    public TransferDetailResponse getTransferDetailByNo(GetTransferDetailByNoRequest request) {
        String requestPath = String.format(GET_DETAIL_BY_BATCH_ID_URL, urlEncode(request.getBatchId()), urlEncode(request.getDetailId()));
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.GET)
                        .url(requestPath)
                        .headers(headers)
                        .build();
        HttpResponse<TransferDetailResponse> httpResponse =
                httpClient.execute(httpRequest, TransferDetailResponse.class);
        return httpResponse.getServiceResponse().cloneWithCipher(decryptor::decrypt);
    }


    /**
     * 通过微信明细单号查询明细单
     *
     * @param request 请求参数
     * @return TransferDetailVo
     * @throws HttpException             发送HTTP请求失败。例如构建请求参数失败、发送请求失败、I/O错误等。包含请求信息。
     * @throws ValidationException       发送HTTP请求成功，验证微信支付返回签名失败。
     * @throws ServiceException          发送HTTP请求成功，服务返回异常。例如返回状态码小于200或大于等于300。
     * @throws MalformedMessageException 服务返回成功，content-type不为application/json、解析返回体失败。
     */
    public TransferDetailResponse getTransferDetailByOutNo(GetTransferDetailByOutNoRequest request) {
        String requestPath = String.format(GET_DETAIL_BY_BATCH_NO_URL, urlEncode(request.getOutBatchNo()), urlEncode(request.getOutDetailNo()));
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.GET)
                        .url(requestPath)
                        .headers(headers)
                        .build();
        HttpResponse<TransferDetailResponse> httpResponse =
                httpClient.execute(httpRequest, TransferDetailResponse.class);
        return httpResponse.getServiceResponse().cloneWithCipher(decryptor::decrypt);
    }

    @Override
    public TransferNotifyDetailResponse doNotify(HttpServletRequest request) {
        String requestBody = HttpServletUtil.getBodyString(request);
        requireNonNull(requestBody);
        // 1. 构造 RequestParam
        RequestParam requestParam = new RequestParam.Builder()
                .serialNumber(request.getHeader(Constant.WECHAT_PAY_SERIAL))  //证书序列号（微信平台）   验签的“微信支付平台证书”所对应的平台证书序列号
                .nonce(request.getHeader(Constant.WECHAT_PAY_NONCE)) //验签的随机字符串
                .signature(request.getHeader(Constant.WECHAT_PAY_SIGNATURE)) //微信传递过来的签名   验签的签名值
                .timestamp(request.getHeader(Constant.WECHAT_PAY_TIMESTAMP)) //验签的时间戳
                .body(requestBody)
                .build();
        // 2. parser验证参数
        try {
            TransferNotifyDetailResponse response = parser.parse(requestParam, TransferNotifyDetailResponse.class);
            return response;
        } catch (Exception e) {
            throw new ValidationException("回调签名验证失败："+e.getMessage());
        }
    }
}
