package com.biz.crm.wechatpay.v3plus.service.impl;

import com.biz.crm.wechatpay.util.HttpServletUtil;
import com.biz.crm.wechatpay.v3plus.model.CancelBillsDetailResponse;
import com.biz.crm.wechatpay.v3plus.model.InitiateBillsTransferRequest;
import com.biz.crm.wechatpay.v3plus.model.InitiateBillsTransferResponse;
import com.biz.crm.wechatpay.v3plus.model.TransferBillsDetailResponse;
import com.biz.crm.wechatpay.v3plus.service.WechatPayV3PlusTransferBatchService;
import com.google.gson.Gson;
import com.wechat.pay.java.core.Config;
import com.wechat.pay.java.core.cipher.PrivacyDecryptor;
import com.wechat.pay.java.core.cipher.PrivacyEncryptor;
import com.wechat.pay.java.core.exception.ValidationException;
import com.wechat.pay.java.core.http.*;
import com.wechat.pay.java.core.notification.NotificationConfig;
import com.wechat.pay.java.core.notification.NotificationParser;
import com.wechat.pay.java.core.notification.RequestParam;

import javax.servlet.http.HttpServletRequest;

import static java.util.Objects.requireNonNull;

/**
 * V3转账新版本
 * 接口文档地址：https://pay.weixin.qq.com/doc/v3/merchant/4012711988
 */
public class WechatPayV3PlusTransferBatchServiceImpl implements WechatPayV3PlusTransferBatchService {
    //根据转账批次号查询转账明细URL
    private static final String GET_TRANSFER_BATCH_BY_BATCH_ID_NEW_URL = "https://api.mch.weixin.qq.com/v3/fund-app/mch-transfer/transfer-bills/transfer-bill-no/%s";
    //根据转账商家单号查询转账明细URL
    private static final String GET_TRANSFER_BATCH_BY_OUT_BATCH_NO_NEW_URL = "https://api.mch.weixin.qq.com/v3/fund-app/mch-transfer/transfer-bills/out-bill-no/%s";
    //发起转账URL
    private static final String TRANSFER_BATCH_NEW_URL = "https://api.mch.weixin.qq.com/v3/fund-app/mch-transfer/transfer-bills";
    //撤销转账
    private static final String TRANSFER_BATCH_CANCEL_NEW_URL = "https://api.mch.weixin.qq.com/v3/fund-app/mch-transfer/transfer-bills/out-bill-no/%s/cancel";


    private final HttpClient httpClient;
    private final PrivacyEncryptor encryptor;
    private final PrivacyDecryptor decryptor;
    private final NotificationParser parser;

    private WechatPayV3PlusTransferBatchServiceImpl(
            HttpClient httpClient,
            PrivacyEncryptor encryptor,
            PrivacyDecryptor decryptor,
            NotificationParser parser) {
        this.httpClient = requireNonNull(httpClient);
        this.encryptor = requireNonNull(encryptor);
        this.decryptor = requireNonNull(decryptor);
        this.parser=requireNonNull(parser);
    }

    /**
     * WechatPayV3TransferBatchService构造器
     */
    public static class Builder {

        private HttpClient httpClient;
        private PrivacyEncryptor encryptor;
        private PrivacyDecryptor decryptor;
        private NotificationParser parser;

        public Builder config(Config config) {
            this.httpClient = new DefaultHttpClientBuilder().config(config).build();
            this.encryptor = config.createEncryptor();
            this.decryptor = config.createDecryptor();
            this.parser= new NotificationParser((NotificationConfig) config);
            return this;
        }

        public Builder httpClient(HttpClient httpClient) {
            this.httpClient = httpClient;
            return this;
        }

        public Builder encryptor(PrivacyEncryptor encryptor) {
            this.encryptor = encryptor;
            return this;
        }

        public Builder decryptor(PrivacyDecryptor decryptor) {
            this.decryptor = decryptor;
            return this;
        }
        public Builder parser(NotificationParser parser) {
            this.parser = parser;
            return this;
        }

        public WechatPayV3PlusTransferBatchService build() {
            return new WechatPayV3PlusTransferBatchServiceImpl(httpClient, encryptor, decryptor,parser);
        }
    }

    @Override
    public InitiateBillsTransferResponse transferBills(InitiateBillsTransferRequest request) {
        if (request.getUserName() != null && !"".equals(request.getUserName())) {
            String encryptName = encryptor.encrypt(request.getUserName());
            request.setUserName(encryptName);
        }
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.WECHAT_PAY_SERIAL, encryptor.getWechatpaySerial());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.POST)
                        .url(TRANSFER_BATCH_NEW_URL)
                        .headers(headers)
                        .body(createRequestBody(request))
                        .build();
        HttpResponse<InitiateBillsTransferResponse> httpResponse = httpClient.execute(httpRequest, InitiateBillsTransferResponse.class);
        return httpResponse.getServiceResponse();
    }

    /**
     * 转json
     *
     * @param request 请求体
     * @return
     */
    private static RequestBody createRequestBody(Object request) {
        return new JsonRequestBody.Builder().body(new Gson().toJson(request)).build();
    }


    @Override
    public TransferBillsDetailResponse getTransferBillsByNo(String transferBillNo) {
        String requestPath = String.format(GET_TRANSFER_BATCH_BY_BATCH_ID_NEW_URL, UrlEncoder.urlEncode(transferBillNo));
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.GET)
                        .url(requestPath)
                        .headers(headers)
                        .build();
        HttpResponse<TransferBillsDetailResponse> httpResponse = httpClient.execute(httpRequest, TransferBillsDetailResponse.class);
        return httpResponse.getServiceResponse().cloneWithCipher(decryptor);
    }

    @Override
    public TransferBillsDetailResponse getTransferBillsByOutNo(String outBillNo) {
        String requestPath = String.format(GET_TRANSFER_BATCH_BY_OUT_BATCH_NO_NEW_URL, UrlEncoder.urlEncode(outBillNo));
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.GET)
                        .url(requestPath)
                        .headers(headers)
                        .build();
        HttpResponse<TransferBillsDetailResponse> httpResponse = httpClient.execute(httpRequest, TransferBillsDetailResponse.class);
        return httpResponse.getServiceResponse().cloneWithCipher(decryptor);
    }

    @Override
    public CancelBillsDetailResponse cancelTransferBills(String outBillNo) {
        String requestPath = String.format(TRANSFER_BATCH_CANCEL_NEW_URL, UrlEncoder.urlEncode(outBillNo));
        HttpHeaders headers = new HttpHeaders();
        headers.addHeader(Constant.ACCEPT, MediaType.APPLICATION_JSON.getValue());
        headers.addHeader(Constant.CONTENT_TYPE, MediaType.APPLICATION_JSON.getValue());
        HttpRequest httpRequest =
                new HttpRequest.Builder()
                        .httpMethod(HttpMethod.POST)
                        .url(requestPath)
                        .headers(headers)
                        .build();
        HttpResponse<CancelBillsDetailResponse> httpResponse = httpClient.execute(httpRequest, CancelBillsDetailResponse.class);
        return httpResponse.getServiceResponse();
    }

    @Override
    public TransferBillsDetailResponse doNotify(HttpServletRequest request) {
        String requestBody = HttpServletUtil.getBodyString(request);
        requireNonNull(requestBody);
        // 1. 构造 RequestParam
        RequestParam requestParam = new RequestParam.Builder()
                .serialNumber(request.getHeader(Constant.WECHAT_PAY_SERIAL))  //证书序列号（微信平台）   验签的“微信支付平台证书”所对应的平台证书序列号
                .nonce(request.getHeader(Constant.WECHAT_PAY_NONCE)) //验签的随机字符串
                .signature(request.getHeader(Constant.WECHAT_PAY_SIGNATURE)) //微信传递过来的签名   验签的签名值
                .timestamp(request.getHeader(Constant.WECHAT_PAY_TIMESTAMP)) //验签的时间戳
                .body(requestBody)
                .build();
        // 2. parser验证参数
        try {
            TransferBillsDetailResponse response = parser.parse(requestParam, TransferBillsDetailResponse.class);
            return response;
        } catch (Exception e) {
            throw new ValidationException("回调签名验证失败："+e.getMessage());
        }
    }
}
